/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal.classloader;

import static java.lang.String.format;
import static java.lang.System.identityHashCode;
import static org.apache.commons.lang3.ClassUtils.getShortClassName;

import java.io.IOException;
import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLClassLoader;
import java.net.URLStreamHandler;
import java.net.URLStreamHandlerFactory;

import sun.net.www.protocol.jar.Handler;

/**
 * {@link URLClassLoader} implementation for Tooling that allows to have a readable {@code toString}.
 *
 * @since 1.0
 */
public class ToolingClassLoader extends URLClassLoader {

  private String muleVersion;
  private String toolingVersion;
  private int bootstrapObjectId;

  /**
   * Creates an instance of this class loader.
   *
   * @param muleVersion {@link String} Mule Runtime version for which this class loader is created.
   * @param toolingVersion {@link String} Tooling Runtime Client version for which this class loader is created.
   * @param bootstrapObjectId identity hash code for bootstrap instance.
   * @param urls {@link URL urls} for the class loader.
   * @param parent {@link ClassLoader parent} to delegate.
   */
  public ToolingClassLoader(String muleVersion, String toolingVersion, int bootstrapObjectId, URL[] urls, ClassLoader parent) {
    super(urls, parent, new NonCachingURLStreamHandlerFactory());
    this.muleVersion = muleVersion;
    this.toolingVersion = toolingVersion;
    this.bootstrapObjectId = bootstrapObjectId;
  }

  @Override
  public String toString() {
    return format("%s{muleVersion=%s, toolingVersion=%s, bootstrapObjectId=%s, objectId=%s}", getShortClassName(this.getClass()),
                  muleVersion,
                  toolingVersion, bootstrapObjectId, identityHashCode(this));
  }

  protected static class NonCachingURLStreamHandlerFactory implements URLStreamHandlerFactory {

    @Override
    public URLStreamHandler createURLStreamHandler(String protocol) {
      return new NonCachingJarResourceURLStreamHandler();
    }
  }

  /**
   * Prevents jar caching for this classloader, mainly to fix the static ResourceBundle mess/cache that keeps connections open no
   * matter what.
   */
  private static class NonCachingJarResourceURLStreamHandler extends Handler {

    public NonCachingJarResourceURLStreamHandler() {
      super();
    }

    @Override
    protected java.net.URLConnection openConnection(URL u) throws IOException {
      JarURLConnection c = new sun.net.www.protocol.jar.JarURLConnection(u, this);
      c.setUseCaches(false);
      return c;
    }
  }

}
