/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal.wrapper;

import static java.util.Objects.requireNonNull;
import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationException;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.artifact.declaration.request.JsonArtifactDeserializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.JsonArtifactSerializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.XmlArtifactDeserializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.XmlArtifactSerializationRequest;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;
import org.mule.tooling.client.internal.serialization.Serializer;

import com.google.common.collect.ImmutableList;

/**
 * Client side implementation for {@link ArtifactSerializationService} that uses reflection and works as a bridge
 * between {@link ClassLoader classLoaders}.
 *
 * @since 1.0
 */
public class ArtifactSerializationServiceWrapper implements ArtifactSerializationService {

  private Dispatcher dispatcher;
  private Serializer serializer;

  public ArtifactSerializationServiceWrapper(Dispatcher dispatcher, Serializer serializer) {
    requireNonNull(dispatcher, "dispatcher cannot be null");
    requireNonNull(serializer, "serializer cannot be null");

    this.dispatcher = dispatcher;
    this.serializer = serializer;
  }

  @Override
  public String toXml(XmlArtifactSerializationRequest request) throws ArtifactSerializationException {
    String result = (String) dispatcher.dispatchRemoteMethod("toXml",
                                                             -1,
                                                             ImmutableList.of(XmlArtifactSerializationRequest.class),
                                                             ImmutableList.of(serializer.serialize(request)));
    return serializer.deserialize(result);
  }

  @Override
  public ArtifactDeclaration fromXml(XmlArtifactDeserializationRequest request) throws ArtifactSerializationException {
    return serializer.deserialize((String) dispatcher.dispatchRemoteMethod("fromXml",
                                                                           -1,
                                                                           ImmutableList
                                                                               .of(XmlArtifactDeserializationRequest.class),
                                                                           ImmutableList.of(serializer.serialize(request))));
  }

  @Override
  public String toJson(JsonArtifactSerializationRequest request) throws ArtifactSerializationException {
    String result = (String) dispatcher.dispatchRemoteMethod("toJson",
                                                             -1,
                                                             ImmutableList.of(JsonArtifactSerializationRequest.class),
                                                             ImmutableList.of(serializer.serialize(request)));
    return serializer.deserialize(result);
  }

  @Override
  public ArtifactDeclaration fromJson(JsonArtifactDeserializationRequest request)
      throws ArtifactSerializationException {
    return serializer.deserialize((String) dispatcher.dispatchRemoteMethod("fromJson",
                                                                           -1,
                                                                           ImmutableList
                                                                               .of(JsonArtifactDeserializationRequest.class),
                                                                           ImmutableList.of(serializer.serialize(request))));
  }

}
