/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.bootstrap.internal;

import static org.hamcrest.CoreMatchers.sameInstance;
import static org.hamcrest.core.IsInstanceOf.instanceOf;
import static org.hamcrest.core.IsNot.not;
import static org.hamcrest.core.IsNull.nullValue;
import static org.junit.Assert.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.when;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.MavenClientProvider;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.runtime.module.embedded.internal.classloading.FilteringClassLoader;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.internal.wrapper.ToolingRuntimeClientBuilderFactoryWrapper;
import org.mule.tooling.client.bootstrap.internal.wrapper.ToolingRuntimeClientBuilderFactoryWrapperBuilder;

import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.net.URLClassLoader;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class DefaultToolingRuntimeClientBootstrapTestCase {

  private static final String MULE_VERSION = "4.2.0";

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();
  @Mock
  private MavenClientProvider mavenClientProvider;
  @Mock
  private MavenClient mavenClient;
  @Mock
  private BundleDependency bundleDependency;
  @Mock
  private ToolingRuntimeClientBuilderFactoryWrapperBuilder toolingRuntimeClientBuilderFactoryWrapperBuilder;
  @Mock
  private ToolingRuntimeClientBuilderFactoryWrapper toolingRuntimeClientBuilderFactoryWrapper;

  @Before
  public void setUp() throws Exception {
    when(bundleDependency.getBundleUri()).thenReturn(temporaryFolder.newFile().toURI());
    when(mavenClient.resolveBundleDescriptor(any(BundleDescriptor.class))).thenReturn(bundleDependency);
    when(mavenClientProvider.createMavenClient(any())).thenReturn(mavenClient);

    when(toolingRuntimeClientBuilderFactoryWrapperBuilder.withToolingVersion(any()))
        .thenReturn(toolingRuntimeClientBuilderFactoryWrapperBuilder);

    when(toolingRuntimeClientBuilderFactoryWrapperBuilder.withToolingClassLoader(any()))
        .thenReturn(toolingRuntimeClientBuilderFactoryWrapperBuilder);
    when(toolingRuntimeClientBuilderFactoryWrapperBuilder.withExecutorService(any()))
        .thenReturn(toolingRuntimeClientBuilderFactoryWrapperBuilder);
    when(toolingRuntimeClientBuilderFactoryWrapperBuilder.withMavenConfiguration(any()))
        .thenReturn(toolingRuntimeClientBuilderFactoryWrapperBuilder);
    when(toolingRuntimeClientBuilderFactoryWrapperBuilder.withWorkingDirectory(any()))
        .thenReturn(toolingRuntimeClientBuilderFactoryWrapperBuilder);

    when(toolingRuntimeClientBuilderFactoryWrapperBuilder.build()).thenReturn(toolingRuntimeClientBuilderFactoryWrapper);
  }

  @Test
  public void createBootstrapWithFileLog4jConfig() throws Exception {
    createToolingBootstrapWithLog4jConfig(this.getClass().getClassLoader().getResource("file-log4j-config.xml").toURI());
  }

  @Test
  public void createBootstrapWithJarLog4jConfig() throws Exception {
    URLClassLoader jarLoadingClassLoader =
        URLClassLoader.newInstance(new URL[] {this.getClass().getClassLoader().getResource("log4j-config.jar")});
    createToolingBootstrapWithLog4jConfig(jarLoadingClassLoader.getResource("jar-log4j-config.xml").toURI());
  }

  private void createToolingBootstrapWithLog4jConfig(URI log4jConfig) throws Exception {

    //Only check that building does not throw any exception
    new DefaultToolingRuntimeClientBootstrap(
                                             ToolingRuntimeClientBootstrapConfiguration.builder()
                                                 .muleVersion(MULE_VERSION)
                                                 .toolingVersion(MULE_VERSION)
                                                 .mavenConfiguration(
                                                                     MavenConfiguration.newMavenConfigurationBuilder()
                                                                         .localMavenRepositoryLocation(
                                                                                                       temporaryFolder.getRoot())
                                                                         .build())
                                                 .log4jConfiguration(log4jConfig)
                                                 .build(),
                                             mavenClientProvider, toolingRuntimeClientBuilderFactoryWrapperBuilder);
  }

  @Test
  public void filterClassLoaderParentClassLoaderShouldNotBeSystemClassLoader() throws IOException {
    ArgumentCaptor<URLClassLoader> toolingClassLoaderArgumentCaptor = ArgumentCaptor.forClass(URLClassLoader.class);

    when(toolingRuntimeClientBuilderFactoryWrapperBuilder.withToolingClassLoader(toolingClassLoaderArgumentCaptor.capture()))
        .thenReturn(toolingRuntimeClientBuilderFactoryWrapperBuilder);


    DefaultToolingRuntimeClientBootstrap toolingRuntimeClientBootstrap =
        new DefaultToolingRuntimeClientBootstrap(ToolingRuntimeClientBootstrapConfiguration.builder()
            .muleVersion("4.2.0").toolingVersion("4.2.0").mavenConfiguration(MavenConfiguration.newMavenConfigurationBuilder()
                .localMavenRepositoryLocation(temporaryFolder.getRoot()).build())
            .build(),
                                                 mavenClientProvider, toolingRuntimeClientBuilderFactoryWrapperBuilder);
    assertThat(toolingRuntimeClientBootstrap, not(nullValue()));
    ClassLoader filteringClassLoader = toolingClassLoaderArgumentCaptor.getValue().getParent();
    assertThat(filteringClassLoader, instanceOf(FilteringClassLoader.class));
    // System ClassLoader on JUnit test cases is the same LauncherAppClassLoader that loaded this class
    assertThat(filteringClassLoader.getParent(), sameInstance(this.getClass().getClassLoader()));
  }

}
