/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal.wrapper;

import static com.google.common.collect.ImmutableList.of;
import static java.util.Objects.requireNonNull;
import org.mule.tooling.client.api.artifact.resources.ResourceLoader;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;
import org.mule.tooling.client.internal.serialization.Serializer;

import java.io.InputStream;
import java.net.URL;
import java.util.Optional;

/**
 * Client side implementation for {@link ResourceLoader} that uses reflection and works as a bridge
 * between {@link ClassLoader classLoaders}.
 *
 * @since 1.2.0, 1.1.4
 */
public class ResourceLoaderWrapper implements ResourceLoader {

  private Dispatcher dispatcher;
  private Serializer serializer;

  public ResourceLoaderWrapper(Dispatcher dispatcher, Serializer serializer) {
    requireNonNull(dispatcher, "dispatcher cannot be null");
    requireNonNull(serializer, "serializer cannot be null");

    this.dispatcher = dispatcher;
    this.serializer = serializer;
  }

  @Override
  public InputStream getResourceAsStream(String resource) {
    return (InputStream) dispatcher.dispatchRemoteMethod("getResourceAsStream", of(String.class), of(resource));
  }

  @Override
  public URL getResource(String resource) {
    return (URL) dispatcher.dispatchRemoteMethod("getResource", of(String.class), of(resource));
  }

  @Override
  public Optional<URL> findIn(String resource, String groupId, String artifactId, Optional<String> version, String classifier,
                              String type) {
    return (Optional<URL>) dispatcher
        .dispatchRemoteMethod("findIn", of(String.class, String.class, String.class, Optional.class, String.class, String.class),
                              of(resource, groupId, artifactId, serializer.serialize(version), classifier, type));
  }

  @Override
  public Optional<InputStream> loadFrom(String resource, String groupId, String artifactId, Optional<String> version,
                                        String classifier, String type) {
    return (Optional<InputStream>) dispatcher
        .dispatchRemoteMethod("loadFrom",
                              of(String.class, String.class, String.class, Optional.class, String.class, String.class),
                              of(resource, groupId, artifactId, serializer.serialize(version), classifier, type));
  }

}
