/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal.wrapper;

import static java.util.Objects.requireNonNull;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolverService;
import org.mule.tooling.client.api.exception.ToolingArtifactNotFoundException;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import org.mule.tooling.client.api.icons.ExtensionIconsService;
import org.mule.tooling.client.api.message.history.MessageHistoryService;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;
import org.mule.tooling.client.internal.serialization.Serializer;

import com.google.common.collect.ImmutableList;

import java.net.URL;
import java.util.Map;

/**
 * Client side implementation for {@link ToolingRuntimeClient} that uses reflection and works as a bridge
 * between {@link ClassLoader classLoaders}.
 *
 * @since 1.0
 */
public class ToolingRuntimeClientWrappper implements ToolingRuntimeClient {

  private Dispatcher dispatcher;
  private Serializer serializer;

  public ToolingRuntimeClientWrappper(Dispatcher dispatcher, Serializer serializer) {
    requireNonNull(dispatcher, "dispatcher cannot be null");
    requireNonNull(serializer, "serializer cannot be null");

    this.dispatcher = dispatcher;
    this.serializer = serializer;
  }

  @Override
  public ToolingArtifact newToolingArtifact(URL artifactUrlContent, Map<String, String> properties) {
    Object toolingArtifactTarget = dispatcher.dispatchRemoteMethod("newToolingArtifact",
                                                                   ImmutableList.of(URL.class, Map.class),
                                                                   ImmutableList.of(serializer.serialize(artifactUrlContent),
                                                                                    serializer.serialize(properties)));
    return new ToolingArtifactWrapper(dispatcher.newReflectionInvoker(toolingArtifactTarget), serializer);
  }

  @Override
  public ToolingArtifact newToolingArtifact(URL artifactUrlContent, Map<String, String> properties, String parentId) {
    Object toolingArtifactTarget = dispatcher.dispatchRemoteMethod("newToolingArtifact",
                                                                   ImmutableList.of(URL.class, Map.class, String.class),
                                                                   ImmutableList.of(serializer.serialize(artifactUrlContent),
                                                                                    serializer.serialize(properties),
                                                                                    serializer.serialize(parentId)));
    return new ToolingArtifactWrapper(dispatcher.newReflectionInvoker(toolingArtifactTarget), serializer);
  }


  @Override
  public ToolingArtifact fetchToolingArtifact(String id) throws ToolingArtifactNotFoundException {
    Object toolingArtifactTarget = dispatcher.dispatchRemoteMethod("fetchToolingArtifact",
                                                                   ImmutableList.of(String.class),
                                                                   ImmutableList.of(serializer.serialize(id)));
    return new ToolingArtifactWrapper(dispatcher.newReflectionInvoker(toolingArtifactTarget), serializer);
  }

  @Override
  public ExtensionModelService extensionModelService() {
    Object extensionModelServiceTarget = dispatcher.dispatchRemoteMethod("extensionModelService");
    return new ExtensionModelServiceWrapper(dispatcher.newReflectionInvoker(extensionModelServiceTarget), serializer);
  }

  @Override
  public ArtifactSerializationService artifactSerializationService() {
    Object artifactSerializationServiceTarget = dispatcher.dispatchRemoteMethod("artifactSerializationService");
    return new ArtifactSerializationServiceWrapper(dispatcher.newReflectionInvoker(artifactSerializationServiceTarget),
                                                   serializer);
  }

  @Override
  public DslSyntaxResolverService dslSyntaxResolverService() {
    Object dslSyntaxResolverServiceTarget = dispatcher.dispatchRemoteMethod("dslSyntaxResolverService");
    return new DslSyntaxResolverServiceWrapper(dispatcher.newReflectionInvoker(dslSyntaxResolverServiceTarget), serializer);
  }

  @Override
  public MessageHistoryService messageHistoryService() {
    Object messageHistoryServiceTarget = dispatcher.dispatchRemoteMethod("messageHistoryService");
    return new MessageHistoryServiceWrapper(dispatcher.newReflectionInvoker(messageHistoryServiceTarget), serializer);
  }

  @Override
  public ExtensionIconsService iconsService() {
    Object iconsServiceTarget = dispatcher.dispatchRemoteMethod("iconsService");
    return new IconsServiceWrapper(dispatcher.newReflectionInvoker(iconsServiceTarget), serializer);
  }

}
