/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal.wrapper;

import static java.util.Objects.requireNonNull;
import static org.mule.tooling.client.api.feature.Feature.disabled;
import static org.mule.tooling.client.api.feature.Feature.enabled;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.exception.ServiceUnavailableException;
import org.mule.tooling.client.api.feature.Action;
import org.mule.tooling.client.api.feature.Feature;
import org.mule.tooling.client.api.metadata.MetadataKeysContainer;
import org.mule.tooling.client.api.metadata.MetadataKeysRequest;
import org.mule.tooling.client.api.metadata.MetadataResult;
import org.mule.tooling.client.api.metadata.MetadataService;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;
import org.mule.tooling.client.internal.serialization.Serializer;

import com.google.common.collect.ImmutableList;

/**
 * Client side implementation for {@link MetadataService} that uses reflection and works as a bridge between {@link ClassLoader
 * classLoaders}.
 *
 * @since 1.0
 */
public class MetadataServiceWrapper implements MetadataService {

  private Dispatcher dispatcher;
  private Serializer serializer;

  public MetadataServiceWrapper(Dispatcher dispatcher, Serializer serializer) {
    requireNonNull(dispatcher, "dispatcher cannot be null");
    requireNonNull(serializer, "serializer cannot be null");

    this.dispatcher = dispatcher;
    this.serializer = serializer;
  }

  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeys(MetadataKeysRequest metadataKeysRequest)
      throws ServiceUnavailableException {
    String result = (String) dispatcher.dispatchRemoteMethod("getMetadataKeys",
                                                             metadataKeysRequest.getRequestTimeout(),
                                                             ImmutableList.of(MetadataKeysRequest.class),
                                                             ImmutableList.of(serializer.serialize(metadataKeysRequest)));
    return serializer.deserialize(result);
  }

  @Override
  public Feature<Action<Location>> disposeMetadataCache() {
    final String methodName = "disposeMetadataCache";
    if (dispatcher.isFeatureEnabled(methodName, new String[0])) {
      Object disposeMetadataCacheTarget = dispatcher.dispatchRemoteMethod(methodName);
      return enabled(new ActionWrapper<>(
                                         dispatcher.newReflectionInvoker(dispatcher
                                             .invokeGetOnFeature(disposeMetadataCacheTarget)),
                                         Location.class, serializer));
    }
    return disabled();
  }

  @Override
  public Feature<Action<Location>> invalidateMetadataCacheKeys() {
    final String methodName = "invalidateMetadataCacheKeys";
    if (dispatcher.isFeatureEnabled(methodName, new String[0])) {
      Object invalidateMetadataCacheKeysTarget = dispatcher.dispatchRemoteMethod(methodName);
      return enabled(new ActionWrapper<>(dispatcher
          .newReflectionInvoker(dispatcher.invokeGetOnFeature(invalidateMetadataCacheKeysTarget)), Location.class, serializer));
    }
    return disabled();
  }
}
