/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.bootstrap.internal;

import static com.google.common.base.Throwables.propagateIfPossible;
import static com.vdurmont.semver4j.Semver.SemverType.STRICT;
import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static java.util.concurrent.TimeUnit.MINUTES;
import static java.util.stream.Collectors.toMap;
import static org.mule.tooling.client.bootstrap.internal.MavenToolingRuntimeClientBootstrap.MULE_RUNTIME_TOOLING_CLIENT;
import static org.mule.tooling.client.bootstrap.internal.MavenToolingRuntimeClientBootstrap.ORG_MULE_TOOLING;
import static org.mule.tooling.client.bootstrap.internal.UpdatePolicyUtils.toMinutes;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.VersionRangeResult;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.extension.model.MuleVersion;
import org.mule.tooling.client.bootstrap.api.DynamicToolingVersionResolverConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingVersionResolver;
import org.mule.tooling.client.bootstrap.api.ToolingVersionResolverConfiguration;
import org.mule.tooling.client.bootstrap.api.UpdatePolicy;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.UncheckedExecutionException;
import com.vdurmont.semver4j.Semver;

import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.concurrent.ExecutionException;

import org.eclipse.aether.util.version.GenericVersionScheme;
import org.eclipse.aether.version.InvalidVersionSpecificationException;
import org.eclipse.aether.version.VersionConstraint;
import org.eclipse.aether.version.VersionScheme;

/**
 * Default implementation of {@link ToolingVersionResolver}.
 */
public class MavenToolingVersionResolver implements ToolingVersionResolver, DynamicToolingVersionResolverConfiguration {

  private static final String SNAPSHOT = "SNAPSHOT";
  private static final String SNAPSHOT_SUFFIX_VERSION = "0-SNAPSHOT";

  private LoadingCache<String, ToolingVersion> cachedRemoteVersions;

  private ToolingVersionResolverConfiguration configuration;
  private MavenClient mavenClient;
  private VersionScheme versionScheme = new GenericVersionScheme();

  public MavenToolingVersionResolver(ToolingVersionResolverConfiguration configuration, MavenClient mavenClient) {
    requireNonNull(configuration, "configuration cannot be null");
    requireNonNull(mavenClient, "mavenClient cannot be null");

    this.configuration = configuration;
    this.mavenClient = mavenClient;

    CacheBuilder<Object, Object> cacheBuilder = CacheBuilder.newBuilder();
    cacheBuilder.refreshAfterWrite(toMinutes(this.configuration.remoteVersionUpdatePolicy()), MINUTES);
    this.cachedRemoteVersions = cacheBuilder.build(new CacheLoader<String, ToolingVersion>() {

      @Override
      public ToolingVersion load(String muleVersion) throws Exception {
        return new DefaultToolingVersion(fetchLatestToolingRuntimeClient(muleVersion), null);
      }

      @Override
      public ListenableFuture<ToolingVersion> reload(String muleVersion,
                                                     ToolingVersion oldValue)
          throws Exception {
        ToolingVersion refreshed = oldValue;
        String newValue = fetchLatestToolingRuntimeClient(muleVersion);
        if (!newValue.equals(oldValue)) {
          refreshed = new DefaultToolingVersion(newValue, oldValue.getVersion());
        }
        return Futures.immediateFuture(refreshed);
      }
    });
  }

  @Override
  public DynamicToolingVersionResolverConfiguration getToolingVersionResolverConfiguration() {
    return this;
  }

  @Override
  public ToolingVersion resolve(String muleVersion) {
    final MuleVersion parsedVersion = new MuleVersion(muleVersion);
    try {
      final Iterator<String> keysIterator = configuration.mappings().keySet().iterator();
      while (keysIterator.hasNext()) {
        String mappedMuleVersion = keysIterator.next();
        VersionConstraint versionConstraint = versionScheme.parseVersionConstraint(mappedMuleVersion);
        if (versionConstraint.containsVersion(versionScheme.parseVersion(muleVersion))) {
          return new DefaultToolingVersion(configuration.mappings().get(mappedMuleVersion), null);
        }
      }

      final int major = parsedVersion.getMajor();
      final int minor = parsedVersion.getMinor();
      return cachedRemoteVersions.get(format("%s.%s", major, minor));
    } catch (InvalidVersionSpecificationException e) {
      throw new ToolingException(format("Error while reading mappings: {}", configuration.mappings()), e);
    } catch (ExecutionException | UncheckedExecutionException e) {
      propagateIfPossible(e.getCause(), IllegalStateException.class);
      throw new ToolingException(format("Error while getting latest version from Maven configuration: {}",
                                        configuration.mavenConfiguration()),
                                 e.getCause());
    }
  }

  private String fetchLatestToolingRuntimeClient(String muleVersion) {
    final String versionRange = toVersionRange(muleVersion);
    final VersionRangeResult versionRangeResult = mavenClient.resolveVersionRange(new BundleDescriptor.Builder()
        .setGroupId(ORG_MULE_TOOLING)
        .setArtifactId(MULE_RUNTIME_TOOLING_CLIENT)
        .setVersion(versionRange)
        .build());
    if (versionRangeResult.getVersions().isEmpty()) {
      throw new IllegalStateException(format("There is no version mapped or available of Tooling Runtime Client for: '%s'",
                                             versionRange));
    }
    ListIterator<String> listIterator = versionRangeResult.getVersions().listIterator(versionRangeResult.getVersions().size());
    while (listIterator.hasPrevious()) {
      Semver semver = new Semver(listIterator.previous(), STRICT);
      if (semver.getSuffixTokens().length == 0) {
        return semver.getOriginalValue();
      }
      if (configuration.allowedSuffixes().containsAll(Arrays.asList(semver.getSuffixTokens()))) {
        return semver.getOriginalValue();
      }
    }
    throw new IllegalStateException(
                                    format("Resolved versions using Maven: '%s' for: '%s' don't match any allowed suffixes: %s",
                                           versionRangeResult.getVersions(), versionRange, configuration.allowedSuffixes()));
  }

  private String toVersionRange(String muleVersion) {
    final MuleVersion parsedVersion = new MuleVersion(muleVersion);
    final int major = parsedVersion.getMajor();
    final int minor = parsedVersion.getMinor();
    return format("[%s.%s.0,%s.%s.%s)", major, minor, major,
                  minor + 1, configuration.allowedSuffixes().contains(SNAPSHOT) ? SNAPSHOT_SUFFIX_VERSION : "0");
  }

  @Override
  public Map<String, String> dynamicMappings() {
    return cachedRemoteVersions.asMap().entrySet().stream()
        .collect(toMap(e -> toVersionRange(e.getKey()), e -> e.getValue().getVersion()));
  }

  @Override
  public MavenConfiguration mavenConfiguration() {
    return configuration.mavenConfiguration();
  }

  @Override
  public UpdatePolicy remoteVersionUpdatePolicy() {
    return configuration.remoteVersionUpdatePolicy();
  }

  @Override
  public List<String> allowedSuffixes() {
    return configuration.allowedSuffixes();
  }

  @Override
  public LinkedHashMap<String, String> mappings() {
    return configuration.mappings();
  }

}
