/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.api;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.bootstrap.internal.DefaultToolingRuntimeClientBootstrapConfigurationBuilder;

import java.io.File;
import java.net.URI;
import java.util.Optional;
import java.util.concurrent.ExecutorService;

/**
 * Configuration for {@link ToolingRuntimeClientBootstrap}.
 *
 * @since 1.0
 */
public interface ToolingRuntimeClientBootstrapConfiguration {

  /**
   * @return the Mule Runtime version for which this bootstrap has been created.
   */
  String muleVersion();

  /**
   * @return the Tooling Runtime Client version to be bootstrapped.
   */
  String toolingVersion();

  /**
   * @return {@link MavenConfiguration} to be used by the {@link ToolingRuntimeClientBootstrapFactory} to resolve the
   *         {@link org.mule.tooling.client.api.ToolingRuntimeClient} implementation version for specified {@link #muleVersion()}.
   */
  MavenConfiguration mavenConfiguration();

  /**
   *
   * @return a {@link File} representing a directory containing the jar files of the Tooling Runtime Client implementation to use.
   *         If {@code null}, the classpath will be calculated based on the Maven dependencies of the
   *         {@code tooling-runtime-client} dependency.
   *
   * @since 1.4
   */
  File toolingLibsFolder();

  /**
   * {@link org.mule.tooling.client.api.ToolingRuntimeClient} runs in an isolated class loader and uses log4j as the logger
   * implementation for slf4j. This configuration will be the one used by log4j during the execution of
   * {@link org.mule.tooling.client.api.ToolingRuntimeClient} code.
   *
   * @return {@link Optional} configuration for log4j. If no one is provided logger will be redirected to {@link System#out}.
   */
  Optional<URI> log4jConfiguration();

  /**
   * @return {@link ExecutorServiceConfiguration} to be used by {@link org.mule.tooling.client.api.ToolingRuntimeClient} in order
   *         to configure a {@link ExecutorService to }submit {@link java.util.concurrent.Callable} tasks to the implementation
   *         {@link ClassLoader}. Triggering task on a different thread will prevent class loader leaks and also allows
   *         {@link org.mule.tooling.client.api.ToolingRuntimeClient} to support time out on operations.
   */
  Optional<ExecutorServiceConfiguration> executorServiceConfiguration();

  /**
   * @return {@link File} to store data needed by Tooling Bootstrap. Should not be shared amount different instances of
   *         Bootstraps. Once the Tooling Bootstrap is disposed the folder would be deleted. It is recommended to avoid using
   *         temporary folders as these folders could be deleted when the OS is suspended, which could generate
   */
  File workingFolder();

  static ToolingRuntimeClientBootstrapConfigurationBuilder builder() {
    return new DefaultToolingRuntimeClientBootstrapConfigurationBuilder();
  }

  /**
   * Builder for {@link ToolingRuntimeClientBootstrapConfiguration}.
   *
   * @since 1.0
   */
  interface ToolingRuntimeClientBootstrapConfigurationBuilder {

    /**
     * @param muleVersion the Mule Runtime version for which this bootstrap has been created.
     */
    ToolingRuntimeClientBootstrapConfigurationBuilder muleVersion(String muleVersion);

    /**
     * @param toolingVersion the Tooling Runtime Client version to be bootstrapped.
     */
    ToolingRuntimeClientBootstrapConfigurationBuilder toolingVersion(String toolingVersion);

    /**
     * This is exclusive with {@link #toolingLibsFolder(File)}.
     *
     * @param mavenConfiguration {@link MavenConfiguration} to be used when resolving Tooling Runtime Client implementation
     *                           version using Maven.
     */
    ToolingRuntimeClientBootstrapConfigurationBuilder mavenConfiguration(MavenConfiguration mavenConfiguration);

    /**
     * This is exclusive with {@link #mavenConfiguration(MavenConfiguration)}.
     *
     * @param toolingLibsFolder {@link File} representing a directory containing the jar files of the Tooling Runtime Client
     *                          implementation to use.
     * @since 1.4
     */
    ToolingRuntimeClientBootstrapConfigurationBuilder toolingLibsFolder(File toolingLibsFolder);

    /**
     * @param log4jConfiguration configuration for log4j. If no one is provided logger will be redirected to {@link System#out}.
     */
    ToolingRuntimeClientBootstrapConfigurationBuilder log4jConfiguration(URI log4jConfiguration);

    /**
     * @param executorServiceConfiguration {@link ExecutorServiceConfiguration} to configure the {@link ExecutorService} to be
     *                                     used by {@link org.mule.tooling.client.api.ToolingRuntimeClient} in order to submit
     *                                     {@link java.util.concurrent.Callable} tasks to the implementation {@link ClassLoader}.
     */
    ToolingRuntimeClientBootstrapConfigurationBuilder executorServiceConfiguration(ExecutorServiceConfiguration executorServiceConfiguration);

    /**
     * @param workingFolder {@link File} to store data needed by Tooling Bootstrap. Should not be shared amount different
     *                      instances of Bootstraps. Once the Tooling Bootstrap is disposed the folder would be deleted. It is
     *                      recommended to avoid using temporary folders as these folders could be deleted when the OS is
     *                      suspended, which could generate issues. If not set a temporary folder will be used.
     */
    ToolingRuntimeClientBootstrapConfigurationBuilder workingFolder(File workingFolder);

    /**
     * @return builds the {@link ToolingRuntimeClientBootstrapConfiguration}.
     */
    ToolingRuntimeClientBootstrapConfiguration build();

  }
}
