/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static org.mule.tooling.client.bootstrap.api.UpdatePolicy.UPDATE_POLICY_DAILY;
import static org.mule.tooling.client.internal.util.Preconditions.checkNotNull;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingVersionResolverConfiguration;
import org.mule.tooling.client.bootstrap.api.UpdatePolicy;

import java.util.LinkedHashMap;
import java.util.List;

/**
 * Default implementation for {@link ToolingVersionResolverConfiguration.ToolingVersionResolverConfigurationBuilder}.
 */
public class DefaultToolingVersionResolverConfigurationBuilder implements
    ToolingVersionResolverConfiguration.ToolingVersionResolverConfigurationBuilder {

  private MavenConfiguration mavenConfiguration;
  private UpdatePolicy updatePolicy;
  private LinkedHashMap<String, String> mappings = new LinkedHashMap<>();
  private List<String> allowedSuffixes = emptyList();

  @Override
  public ToolingVersionResolverConfiguration.ToolingVersionResolverConfigurationBuilder mavenConfiguration(
                                                                                                           MavenConfiguration mavenConfiguration) {
    checkNotNull(mavenConfiguration, "mavenConfiguration cannot be null");
    this.mavenConfiguration = mavenConfiguration;
    return this;
  }

  @Override
  public ToolingVersionResolverConfiguration.ToolingVersionResolverConfigurationBuilder mappings(LinkedHashMap mappings) {
    checkNotNull(mappings, "mappings cannot be null");
    this.mappings.clear();
    this.mappings.putAll(mappings);
    return this;
  }

  @Override
  public ToolingVersionResolverConfiguration.ToolingVersionResolverConfigurationBuilder updatePolicy(UpdatePolicy updatePolicy) {
    checkNotNull(updatePolicy, "updatePolicy cannot be null");
    this.updatePolicy = updatePolicy;
    return this;
  }

  @Override
  public ToolingVersionResolverConfiguration.ToolingVersionResolverConfigurationBuilder allowedSuffixes(
                                                                                                        String... allowedSuffixes) {
    checkNotNull(allowedSuffixes, "allowedSuffixes cannot be null");
    this.allowedSuffixes = asList(allowedSuffixes);
    return this;
  }

  @Override
  public ToolingVersionResolverConfiguration build() {
    return new DefaultToolingRuntimeClientBootstrapConfiguration(mavenConfiguration, updatePolicy, mappings, allowedSuffixes);
  }

  private static class DefaultToolingRuntimeClientBootstrapConfiguration implements ToolingVersionResolverConfiguration {

    private MavenConfiguration mavenConfiguration;
    private UpdatePolicy updatePolicy;
    private LinkedHashMap<String, String> mappings = new LinkedHashMap<>();
    private List<String> allowedSuffixes;

    DefaultToolingRuntimeClientBootstrapConfiguration(MavenConfiguration mavenConfiguration, UpdatePolicy updatePolicy,
                                                      LinkedHashMap<String, String> mappings, List<String> allowedSuffixes) {
      checkNotNull(mavenConfiguration, "mavenConfiguration cannot be null");
      this.mavenConfiguration = mavenConfiguration;
      this.updatePolicy = updatePolicy == null ? new UpdatePolicy(UPDATE_POLICY_DAILY) : updatePolicy;
      this.mappings.putAll(mappings);
      this.allowedSuffixes = allowedSuffixes;
    }

    @Override
    public MavenConfiguration mavenConfiguration() {
      return mavenConfiguration;
    }

    @Override
    public UpdatePolicy remoteVersionUpdatePolicy() {
      return updatePolicy;
    }

    @Override
    public LinkedHashMap<String, String> mappings() {
      return mappings;
    }

    @Override
    public List<String> allowedSuffixes() {
      return allowedSuffixes;
    }

  }

}
