/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal.wrapper;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.ServiceLoader.load;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.ToolingRuntimeClientBuilderFactory;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;
import org.mule.tooling.client.internal.serialization.KryoClientSerializer;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.client.internal.serialization.XStreamClientSerializer;

import java.io.File;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Client side implementation for {@link ToolingRuntimeClientBuilderFactory} that uses reflection and works as a bridge between
 * {@link ClassLoader classLoaders}.
 *
 * @since 1.0
 */
public class ToolingRuntimeClientBuilderFactoryWrapper implements ToolingRuntimeClientBuilderFactory, Disposable {

  public static final String ORG_MULE_TOOLING_CLIENT_INTERNAL_DEFAULT_TOOLING_RUNTIME_CLIENT_FACTORY =
      "org.mule.tooling.client.internal.DefaultToolingRuntimeClientFactory";
  private static Logger LOGGER = LoggerFactory.getLogger(ToolingRuntimeClientBuilderFactoryWrapper.class);

  private String toolingVersion;
  private File workingDirectory;
  private ClassLoader toolingClassLoader;
  private Dispatcher dispatcher;
  private Serializer serializer;
  private MavenConfiguration mavenConfiguration;

  public ToolingRuntimeClientBuilderFactoryWrapper(String toolingVersion, ClassLoader toolingClassLoader,
                                                   ExecutorService executorService, MavenConfiguration mavenConfiguration,
                                                   File workingDirectory) {
    requireNonNull(toolingVersion, "toolingVersion cannot be null");
    requireNonNull(toolingClassLoader, "toolingClassLoader cannot be null");
    requireNonNull(executorService, "executorService cannot be null");
    requireNonNull(mavenConfiguration, "mavenConfiguration cannot be null");
    requireNonNull(workingDirectory, "workingDirectory cannot be null");

    this.toolingVersion = toolingVersion;
    this.workingDirectory = workingDirectory;
    this.toolingClassLoader = toolingClassLoader;

    Object toolingRuntimeClientFactory = withContextClassLoader(toolingClassLoader, () -> discoverToolingRuntimeClientFactory());
    this.serializer = discoverSerialization(toolingRuntimeClientFactory);
    this.dispatcher = new Dispatcher(toolingRuntimeClientFactory, toolingClassLoader, executorService);
    this.mavenConfiguration = mavenConfiguration;

    withContextClassLoader(toolingClassLoader, () -> initialise(toolingClassLoader, toolingRuntimeClientFactory));
  }

  private void withContextClassLoader(ClassLoader toolingClassLoader, Runnable runnable) {
    withContextClassLoader(toolingClassLoader, () -> {
      runnable.run();
      return null;
    });
  }

  private <T> T withContextClassLoader(ClassLoader toolingClassLoader, Callable<T> callable) {
    ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
    try {
      Thread.currentThread().setContextClassLoader(toolingClassLoader);
      return callable.call();
    } catch (ToolingException e) {
      throw e;
    } catch (Exception e) {
      throw new ToolingException(e);
    } finally {
      Thread.currentThread().setContextClassLoader(contextClassLoader);
    }
  }

  private void initialise(ClassLoader toolingClassLoader, Object toolingRuntimeClientFactory) {
    ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
    try {
      Thread.currentThread().setContextClassLoader(toolingClassLoader);
      toolingRuntimeClientFactory.getClass().getMethod("initialise").invoke(toolingRuntimeClientFactory);
    } catch (NoSuchMethodException e) {
    } catch (IllegalAccessException e) {
      throw new ToolingException("Error while initializing Tooling Runtime Client factory", e);
    } catch (InvocationTargetException e) {
      throw new ToolingException("Error while initializing Tooling Runtime Client factory", e);
    } finally {
      Thread.currentThread().setContextClassLoader(contextClassLoader);
    }
  }

  private Serializer discoverSerialization(Object toolingRuntimeClientFactory) {
    try {
      toolingRuntimeClientFactory.getClass().getMethod("setSerialization", String.class).invoke(toolingRuntimeClientFactory,
                                                                                                KryoClientSerializer.NAME);
      LOGGER.info("Using Kryo serialization");
      return new KryoClientSerializer(this.getClass().getClassLoader(), toolingRuntimeClientFactory.getClass().getClassLoader());
    } catch (NoSuchMethodException e) {
      // Kryo is not supported by the bootstrapped instance
      LOGGER.info("Using xStream serialization, as API supports Kryo but not the bootstrapped implementation");
      return new XStreamClientSerializer(toolingRuntimeClientFactory.getClass().getClassLoader());
    } catch (IllegalAccessException e) {
      throw new ToolingException("Error while discovering serialization", e);
    } catch (InvocationTargetException e) {
      throw new ToolingException("Error while discovering serialization", e);
    }
  }

  private Object discoverToolingRuntimeClientFactory() {
    Class<?> defaultToolingRuntimeClientFactory;
    try {
      defaultToolingRuntimeClientFactory =
          toolingClassLoader.loadClass(ORG_MULE_TOOLING_CLIENT_INTERNAL_DEFAULT_TOOLING_RUNTIME_CLIENT_FACTORY);
    } catch (ClassNotFoundException e) {
      throw new IllegalStateException(format(
                                             "ClassLoader for Tooling was not correctly created as there is no implementation of %s",
                                             ToolingRuntimeClientBuilderFactory.class.getName()),
                                      e);

    }

    try {
      // 4.1.5 added support for working directory
      Optional<Constructor> constructor =
          getConstructor(defaultToolingRuntimeClientFactory, String.class, File.class);
      if (constructor.isPresent()) {
        return constructor.get().newInstance(toolingVersion, workingDirectory);
      }

      constructor = getConstructor(defaultToolingRuntimeClientFactory, String.class);
      if (constructor.isPresent()) {
        // Tooling Runtime Client 4.1 added support for validation of minMuleVersion (correctly) based on changes on Bootstrapping
        return toolingClassLoader.loadClass(ORG_MULE_TOOLING_CLIENT_INTERNAL_DEFAULT_TOOLING_RUNTIME_CLIENT_FACTORY)
            .getConstructor(String.class).newInstance(toolingVersion);
      }

      // Tooling Runtime Client 4.0 would keep on working with SPI and use Mule Version from Mule Core jar manifest
      ServiceLoader<? extends Object> serviceLoader =
          load(toolingClassLoader.loadClass(ToolingRuntimeClientBuilderFactory.class.getName()), toolingClassLoader);
      if (!serviceLoader.iterator().hasNext()) {
        throw new IllegalStateException("No service found for: '" + ToolingRuntimeClientBuilderFactory.class.getName()
            + "'");
      }
      return serviceLoader.iterator().next();
    } catch (Exception e) {
      throw new IllegalStateException(format(
                                             "Error while creating the implementation of %s",
                                             ToolingRuntimeClientBuilderFactory.class.getName()),
                                      e);
    }

  }

  private static Optional<Constructor> getConstructor(Class clazz, Class... parameterTypes) {
    try {
      return of(clazz.getConstructor(parameterTypes));
    } catch (NoSuchMethodException e) {
      return empty();
    }
  }

  @Override
  public void dispose() {
    dispatcher.dispatchRemoteMethod("dispose");
  }

  @Override
  public ToolingRuntimeClient.Builder create() {
    Object builderTarget = dispatcher.dispatchRemoteMethod("create");
    ToolingRuntimeClient.Builder builder =
        new BuilderWrapper(toolingClassLoader, dispatcher.newReflectionInvoker(builderTarget), serializer);
    // By default it mavenConfiguration from bootstrap is inherited
    builder.withMavenConfiguration(mavenConfiguration);
    return builder;
  }

  public static ToolingRuntimeClientBuilderFactoryWrapperBuilder builder() {
    return new DefaultToolingRuntimeClientBuilderFactoryWrapper();
  }

  private static class DefaultToolingRuntimeClientBuilderFactoryWrapper
      implements ToolingRuntimeClientBuilderFactoryWrapperBuilder {

    private String toolingVersion;
    private ClassLoader toolingClassLoader;
    private ExecutorService executorService;
    private MavenConfiguration mavenConfiguration;
    private File workingDirectory;

    @Override
    public ToolingRuntimeClientBuilderFactoryWrapperBuilder withToolingVersion(String toolingVersion) {
      requireNonNull(toolingVersion, "toolingVersion cannot be null");
      this.toolingVersion = toolingVersion;
      return this;
    }

    @Override
    public ToolingRuntimeClientBuilderFactoryWrapperBuilder withToolingClassLoader(ClassLoader toolingClassLoader) {
      requireNonNull(toolingClassLoader, "toolingClassLoader cannot be null");
      this.toolingClassLoader = toolingClassLoader;
      return this;
    }

    @Override
    public ToolingRuntimeClientBuilderFactoryWrapperBuilder withExecutorService(ExecutorService executorService) {
      requireNonNull(executorService, "executorService cannot be null");
      this.executorService = executorService;
      return this;
    }

    @Override
    public ToolingRuntimeClientBuilderFactoryWrapperBuilder withMavenConfiguration(MavenConfiguration mavenConfiguration) {
      requireNonNull(mavenConfiguration, "mavenConfiguration cannot be null");
      this.mavenConfiguration = mavenConfiguration;
      return this;
    }

    @Override
    public ToolingRuntimeClientBuilderFactoryWrapperBuilder withWorkingDirectory(File workingDirectory) {
      requireNonNull(workingDirectory, "workingDirectory cannot be null");
      this.workingDirectory = workingDirectory;
      return this;
    }

    @Override
    public ToolingRuntimeClientBuilderFactoryWrapper build() {
      return new ToolingRuntimeClientBuilderFactoryWrapper(toolingVersion, toolingClassLoader, executorService,
                                                           mavenConfiguration, workingDirectory);
    }
  }
}
