/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
/*
* Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
* The software in this package is published under the terms of the CPAL v1.0
* license, a copy of which has been included with this distribution in the
* LICENSE.txt file.
*/

package org.mule.tooling.client.bootstrap.internal.wrapper;

import static java.util.Objects.requireNonNull;
import static org.mule.tooling.client.api.feature.Feature.disabled;
import static org.mule.tooling.client.api.feature.Feature.enabled;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.cache.CacheStorageFactory;
import org.mule.tooling.client.api.cache.CacheStorageSerializer;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.datasense.MetadataCacheFactory;
import org.mule.tooling.client.api.datasense.storage.MetadataCacheStorageSerializer;
import org.mule.tooling.client.api.datasense.storage.MetadataCacheStorageFactory;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.feature.Action;
import org.mule.tooling.client.api.feature.Feature;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;
import org.mule.tooling.client.internal.cache.CacheStorageMapWrapperFactory;
import org.mule.tooling.client.internal.metadata.MetadataCacheStorageMapWrapperFactory;
import org.mule.tooling.client.internal.serialization.Serializer;

import com.google.common.collect.ImmutableList;

import java.lang.reflect.Method;

/**
 * Client side implementation for {@link ToolingRuntimeClient} that uses reflection and works as a bridge between
 * {@link ClassLoader classLoaders}.
 *
 * @since 1.0
 */
public class BuilderWrapper implements ToolingRuntimeClient.Builder {

  private final Dispatcher dispatcher;
  private ClassLoader toolingClassLoader;
  private Serializer serializer;

  public BuilderWrapper(ClassLoader toolingClassLoader, Dispatcher dispatcher, Serializer serializer) {
    requireNonNull(dispatcher, "dispatcher cannot be null");
    requireNonNull(toolingClassLoader, "toolingClassLoader cannot be null");
    requireNonNull(serializer, "serializer cannot be null");

    this.dispatcher = dispatcher;
    this.toolingClassLoader = toolingClassLoader;
    this.serializer = serializer;
  }

  @Override
  public ToolingRuntimeClient.Builder withRemoteAgentConfiguration(AgentConfiguration agentConfiguration) {
    dispatcher.dispatchRemoteMethod("withRemoteAgentConfiguration",
                                    ImmutableList.of(AgentConfiguration.class),
                                    ImmutableList.of(serializer.serialize(agentConfiguration)));
    return this;
  }

  @Override
  public ToolingRuntimeClient.Builder withMavenConfiguration(MavenConfiguration mavenConfiguration) {
    dispatcher.dispatchRemoteMethod("withMavenConfiguration",
                                    ImmutableList.of(MavenConfiguration.class),
                                    ImmutableList.of(serializer.serialize(mavenConfiguration)));
    return this;
  }

  @Override
  public Feature<Action<String>> withTargetRuntimeVersion() {
    String withTargetRuntimeVersionMethodName = "withTargetRuntimeVersion";
    if (dispatcher.isFeatureEnabled(withTargetRuntimeVersionMethodName, new String[0])) {
      Object withTargetRuntimeVersionTarget = dispatcher.dispatchRemoteMethod(withTargetRuntimeVersionMethodName);
      return Feature.enabled(targetMuleRuntimeVersion -> new ActionWrapper<>(
                                                                             dispatcher.newReflectionInvoker(dispatcher
                                                                                 .invokeGetOnFeature(withTargetRuntimeVersionTarget)),
                                                                             String.class, serializer)
                                                                                 .perform(targetMuleRuntimeVersion));
    }
    return disabled();
  }

  @Override
  public ToolingRuntimeClient.Builder withMetadataCacheFactory(MetadataCacheFactory metadataCacheFactory) {
    // Different implementation as it has to create a proxy on implementation side that is able to invoke methods on the provided
    // implementation.
    try {
      Class proxyMetadataCacheFactoryClazz =
          toolingClassLoader.loadClass("org.mule.tooling.client.internal.MetadataCacheFactoryProxy");
      String methodName = "withMetadataCacheFactory";
      Object proxyMetadataCacheFactory =
          proxyMetadataCacheFactoryClazz.getConstructor(Object.class).newInstance(metadataCacheFactory);
      Method method = dispatcher.findMethod(methodName);
      dispatcher.invoke(method, -1, new Object[] {proxyMetadataCacheFactory});
      return this;
    } catch (Exception e) {
      throw new ToolingException("Couldn't create proxy for MetadataCacheFactory", e);
    }
  }

  @Override
  public Feature<Action<MetadataCacheStorageFactory>> withMetadataCacheStorageFactory() {
    // First, try to return the logic to configure the metadata cache the new way. If enabled, it means that we are
    // using a tooling-client with this logic available. Otherwise, use the old way.
    Feature<Action<MetadataCacheStorageFactory>> newAction =
        configureCacheStorageFactory("withCacheStorageFactoryForMetadata", "withCacheStorageMapWrapperFactoryForMetadata");

    if (newAction.isEnabled()) {
      return newAction;
    }
    return configureDeprecatedMetadataCacheStorage();
  }

  @Override
  public Feature<Action<CacheStorageFactory>> withCacheStorageFactoryForMetadata() {
    final String builderMethod = "withCacheStorageFactoryForMetadata";
    final String setterMethod = "withCacheStorageMapWrapperFactoryForMetadata";
    Feature<Action<CacheStorageFactory>> feature = configureCacheStorageFactory(builderMethod, setterMethod);
    if (feature.isEnabled()) {
      return feature;
    }

    // Check if the old way is available, otherwise, return disabled.
    return configureDeprecatedMetadataCacheStorage();
  }

  @Override
  public Feature<Action<CacheStorageFactory>> withCacheStorageFactoryForValueProviders() {
    final String builderMethod = "withCacheStorageFactoryForValueProviders";
    final String setterMethod = "withCacheStorageMapWrapperFactoryForValueProviders";
    return configureCacheStorageFactory(builderMethod, setterMethod);
  }

  private <T extends CacheStorageFactory> Feature<Action<T>> configureDeprecatedMetadataCacheStorage() {
    final String withMetadataCacheStorageFactoryMethodName = "withMetadataCacheStorageFactory";
    if (dispatcher.isFeatureEnabled(withMetadataCacheStorageFactoryMethodName, new String[0])) {

      return enabled(
                     metadataCacheStorageFactory -> {
                       try {
                         Class defaultMetadataCacheStorageSerializerClass = toolingClassLoader
                             .loadClass("org.mule.tooling.client.internal.metadata.DefaultMetadataCacheStorageSerializer");
                         Object cacheSerializer = defaultMetadataCacheStorageSerializerClass.getConstructor().newInstance();
                         MetadataCacheStorageSerializer metadataCacheStorageSerializer =
                             new CacheStorageSerializerProxy(cacheSerializer);
                         MetadataCacheStorageMapWrapperFactory mapWrapperFactory =
                             new DefaultCacheStorageMapWrapperFactory(metadataCacheStorageFactory,
                                                                      metadataCacheStorageSerializer);
                         Class proxyMetadataCacheStorageMapWrapperFactoryClazz =
                             toolingClassLoader
                                 .loadClass("org.mule.tooling.client.internal.metadata.MetadataCacheStorageMapWrapperFactoryProxy");
                         String methodName = "withMetadataCacheStorageMapWrapperFactory";
                         Object proxyMetadataCacheFactory =
                             proxyMetadataCacheStorageMapWrapperFactoryClazz.getConstructor(Object.class)
                                 .newInstance(mapWrapperFactory);
                         Method method = dispatcher.findMethod(methodName);
                         dispatcher.invoke(method, -1, new Object[] {proxyMetadataCacheFactory});
                       } catch (Exception e) {
                         throw new ToolingException("Couldn't create proxy for MetadataCacheFactory", e);
                       }
                     });
    }
    return disabled();
  }


  private <T extends CacheStorageFactory> Feature<Action<T>> configureCacheStorageFactory(String builderMethod,
                                                                                          String setterMethod) {
    if (dispatcher.isFeatureEnabled(builderMethod, new String[0])) {
      return enabled(
                     metadataCacheStorageFactory -> {
                       try {
                         CacheStorageSerializer cacheStorageSerializer =
                             CacheStorageSerializerProxy.withDefaultTarget(toolingClassLoader);
                         CacheStorageMapWrapperFactory mapWrapperFactory =
                             new DefaultCacheStorageMapWrapperFactory(metadataCacheStorageFactory, cacheStorageSerializer);
                         Class proxyCacheStorageMapWrapperFactoryClazz = toolingClassLoader
                             .loadClass("org.mule.tooling.client.internal.cache.CacheStorageMapWrapperFactoryProxy");
                         Object proxyCacheFactory =
                             proxyCacheStorageMapWrapperFactoryClazz.getConstructor(Object.class).newInstance(mapWrapperFactory);
                         Method method = dispatcher.findMethod(setterMethod);
                         dispatcher.invoke(method, -1, new Object[] {proxyCacheFactory});
                       } catch (Exception e) {
                         throw new ToolingException("Couldn't create proxy for MetadataCacheFactory", e);
                       }
                     });
    }
    return disabled();
  }

  @Override
  public ToolingRuntimeClient build() {
    Object clientTarget = dispatcher.dispatchRemoteMethod("build");
    return new ToolingRuntimeClientWrappper(dispatcher.newReflectionInvoker(clientTarget),
                                            serializer,
                                            toolingClassLoader);
  }

}
