/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.bootstrap.internal.wrapper;

import static java.util.Arrays.asList;
import static java.util.Objects.requireNonNull;

import org.mule.runtime.app.declaration.api.ComponentElementDeclaration;
import org.mule.runtime.app.declaration.api.ConfigurationElementDeclaration;
import org.mule.runtime.app.declaration.api.ParameterizedElementDeclaration;
import org.mule.runtime.app.declaration.api.TopLevelParameterDeclaration;
import org.mule.tooling.client.api.cache.CacheStorage;
import org.mule.tooling.client.api.connectivity.ConnectionValidationResult;
import org.mule.tooling.client.api.declaration.session.DeclarationSession;
import org.mule.tooling.client.api.declaration.session.DeclarationSessionCacheService;
import org.mule.tooling.client.api.descriptors.dependency.Dependency;
import org.mule.tooling.client.api.metadata.ComponentMetadataTypesDescriptor;
import org.mule.tooling.client.api.metadata.MetadataResult;
import org.mule.tooling.client.api.sampledata.ComponentSampleDataResult;
import org.mule.tooling.client.api.value.resolver.ValueResolverResult;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;
import org.mule.tooling.client.internal.serialization.Serializer;

import java.lang.reflect.Method;
import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableList;


public class DeclarationSessionWrapper implements DeclarationSession {

  private Dispatcher dispatcher;
  private Serializer serializer;

  public DeclarationSessionWrapper(Dispatcher dispatcher,
                                   Serializer serializer) {
    requireNonNull(dispatcher, "dispatcher cannot be null");
    requireNonNull(serializer, "serializer cannot be null");
    requireNonNull(serializer, "cache service cannot be null");

    this.dispatcher = dispatcher;
    this.serializer = serializer;
  }

  @Override
  public ValueResolverResult getValues(ParameterizedElementDeclaration elementDeclaration,
                                       String parameterName,
                                       boolean ignoreCache) {
    String result = (String) dispatcher.dispatchRemoteMethod("getValues",
                                                             ImmutableList.of(ParameterizedElementDeclaration.class,
                                                                              String.class,
                                                                              Boolean.class),
                                                             ImmutableList.of(
                                                                              serializer.serialize(elementDeclaration),
                                                                              serializer.serialize(parameterName),
                                                                              serializer.serialize(ignoreCache)));
    return serializer.deserialize(result);
  }

  @Override
  public ValueResolverResult getFieldValues(ParameterizedElementDeclaration elementDeclaration,
                                            String parameterName, String targetPath,
                                            boolean ignoreCache) {
    String result = (String) dispatcher.dispatchRemoteMethod("getFieldValues",
                                                             ImmutableList.of(ParameterizedElementDeclaration.class,
                                                                              String.class, String.class,
                                                                              Boolean.class),
                                                             ImmutableList.of(
                                                                              serializer.serialize(elementDeclaration),
                                                                              serializer.serialize(parameterName),
                                                                              serializer.serialize(targetPath),
                                                                              serializer.serialize(ignoreCache)));
    return serializer.deserialize(result);
  }

  @Override
  public MetadataResult<ComponentMetadataTypesDescriptor> resolveComponentMetadata(ComponentElementDeclaration componentElementDeclaration,
                                                                                   boolean ignoreCache) {
    String result = (String) dispatcher.dispatchRemoteMethod("resolveComponentMetadata",
                                                             ImmutableList.of(ComponentElementDeclaration.class, Boolean.class),
                                                             ImmutableList.of(serializer.serialize(componentElementDeclaration),
                                                                              serializer.serialize(ignoreCache)));
    return serializer.deserialize(result);
  }

  @Override
  public ComponentSampleDataResult getSampleData(ComponentElementDeclaration componentElementDeclaration) {
    String result = (String) dispatcher.dispatchRemoteMethod("getSampleData",
                                                             ImmutableList.of(ComponentElementDeclaration.class),
                                                             ImmutableList.of(serializer.serialize(componentElementDeclaration)));
    return serializer.deserialize(result);
  }

  @Override
  public ConnectionValidationResult testConnection(String configName) {
    String result = (String) dispatcher.dispatchRemoteMethod("testConnection",
                                                             ImmutableList.of(String.class),
                                                             ImmutableList.of(serializer.serialize(configName)));
    return serializer.deserialize(result);
  }

  @Override
  public DeclarationSessionCacheService getCacheService() {
    return new DeclarationSessionCacheServiceWrapper(
                                                     dispatcher.newReflectionInvoker(dispatcher
                                                         .dispatchRemoteMethod("getCacheService")),
                                                     serializer);
  }

  @Override
  public void dispose() {
    dispatcher.dispatchRemoteMethod("dispose");
  }

  public static class Builder implements DeclarationSession.Builder {

    private final Dispatcher dispatcher;
    private final Serializer serializer;

    public Builder(Dispatcher dispatcher,
                   Serializer serializer,
                   ClassLoader toolingClassLoader) {
      requireNonNull(dispatcher, "dispatcher cannot be null");
      requireNonNull(serializer, "serializer cannot be null");
      requireNonNull(toolingClassLoader, "classloader cannot be null");

      this.dispatcher = dispatcher;
      this.serializer = serializer;
    }

    @Override
    public DeclarationSession.Builder withConfigurationElementDeclarations(List<ConfigurationElementDeclaration> configurationElementDeclarations) {
      dispatcher.dispatchRemoteMethod("withConfigurationElementDeclarations", asList(List.class),
                                      asList(serializer.serialize(configurationElementDeclarations)));
      return this;
    }

    @Override
    public DeclarationSession.Builder withGlobalParameters(List<TopLevelParameterDeclaration> globalParameterDeclarations) {
      dispatcher.dispatchRemoteMethod("withGlobalParameters", asList(List.class),
                                      asList(serializer.serialize(globalParameterDeclarations)));
      return this;
    }

    @Override
    public DeclarationSession.Builder withSessionProperties(Map<String, String> sessionProperties) {
      dispatcher.dispatchRemoteMethod("withSessionProperties", asList(Map.class),
                                      asList(serializer.serialize(sessionProperties)));
      return this;
    }

    @Override
    public DeclarationSession.Builder withDependency(Dependency dependency) {
      dispatcher.dispatchRemoteMethod("withDependency", asList(Dependency.class), asList(serializer.serialize(dependency)));
      return this;
    }

    @Override
    public DeclarationSession.Builder withCacheStorage(CacheStorage cacheStorage) {
      Map<String, Object> storageMap = new CacheStorageMapWrapper(cacheStorage);
      // Call directly to avoid serialization
      Method method = dispatcher.findMethod("internalWithCacheStorage");
      dispatcher.invoke(method, -1, new Object[] {storageMap});
      return this;
    }

    @Override
    public DeclarationSession build() {
      return new DeclarationSessionWrapper(dispatcher.newReflectionInvoker(dispatcher.dispatchRemoteMethod("build")),
                                           serializer);
    }
  }
}
