/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal.wrapper;

import static java.util.Objects.requireNonNull;
import static org.mule.tooling.client.api.feature.Feature.disabled;
import static org.mule.tooling.client.api.feature.Feature.enabled;

import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.artifact.ast.ArtifactAst;
import org.mule.tooling.client.api.artifact.resources.ResourceLoader;
import org.mule.tooling.client.api.component.location.ComponentLocationService;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingService;
import org.mule.tooling.client.api.datasense.DataSenseService;
import org.mule.tooling.client.api.dataweave.DataWeaveService;
import org.mule.tooling.client.api.feature.Feature;
import org.mule.tooling.client.api.metadata.MetadataService;
import org.mule.tooling.client.api.sampledata.SampleDataService;
import org.mule.tooling.client.api.value.provider.ValueProviderService;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;
import org.mule.tooling.client.internal.serialization.Serializer;

import java.util.Map;
import java.util.Optional;

/**
 * Client side implementation for {@link ToolingArtifact} that uses reflection and works as a bridge between {@link ClassLoader
 * classLoaders}.
 *
 * @since 1.0
 */
public class ToolingArtifactWrapper implements ToolingArtifact {

  private Dispatcher dispatcher;
  private Serializer serializer;

  public ToolingArtifactWrapper(Dispatcher dispatcher, Serializer serializer) {
    requireNonNull(dispatcher, "dispatcher cannot be null");
    requireNonNull(serializer, "serializer cannot be null");

    this.dispatcher = dispatcher;
    this.serializer = serializer;
  }

  @Override
  public Optional<ToolingArtifact> getParent() {
    Optional parentTarget = (Optional) dispatcher.dispatchRemoteMethod("getParent");
    return parentTarget.map(target -> new ToolingArtifactWrapper(dispatcher.newReflectionInvoker(target), serializer));
  }

  @Override
  public String getId() {
    return serializer.deserialize((String) dispatcher.dispatchRemoteMethod("id"));
  }

  @Override
  public Map<String, String> getProperties() {
    return serializer.deserialize((String) dispatcher.dispatchRemoteMethod("getProperties"));
  }

  @Override
  public ConnectivityTestingService connectivityTestingService() {
    Object connectivityTestingServiceTarget = dispatcher.dispatchRemoteMethod("connectivityTestingService");
    return new ConnectivityTestingServiceWrapper(dispatcher.newReflectionInvoker(connectivityTestingServiceTarget), serializer);
  }

  @Override
  public MetadataService metadataService() {
    Object metadataServiceTarget = dispatcher.dispatchRemoteMethod("metadataService");
    return new MetadataServiceWrapper(dispatcher.newReflectionInvoker(metadataServiceTarget), serializer);
  }

  @Override
  public DataSenseService dataSenseService() {
    Object dataSenseServiceTarget = dispatcher.dispatchRemoteMethod("dataSenseService");
    return new DataSenseServiceWrapper(dispatcher.newReflectionInvoker(dataSenseServiceTarget), serializer);
  }

  @Override
  public DataWeaveService dataWeaveService() {
    Object dataWeaveServiceTarget = dispatcher.dispatchRemoteMethod("dataWeaveService");
    return new DataWeaveServiceWrapper(dispatcher.newReflectionInvoker(dataWeaveServiceTarget), serializer);
  }

  @Override
  public ValueProviderService valueProviderService() {
    Object valueProviderServiceTarget = dispatcher.dispatchRemoteMethod("valueProviderService");
    return new ValueProviderServiceWrapper(dispatcher.newReflectionInvoker(valueProviderServiceTarget), serializer);
  }

  @Override
  public Feature<SampleDataService> sampleDataService() {
    final String methodName = "sampleDataService";
    if (dispatcher.isFeatureEnabled(methodName, new String[0])) {
      Object sampleDataServiceTarget = dispatcher.dispatchRemoteMethod("sampleDataService");
      return enabled(new SampleDataServiceWrapper(dispatcher
          .newReflectionInvoker(dispatcher.invokeGetOnFeature(sampleDataServiceTarget)), serializer));
    }
    return disabled();
  }

  @Override
  public Feature<ResourceLoader> getResourceLoader() {
    if (dispatcher.isFeatureEnabled("getResourceLoader", new String[0])) {
      Object resourceLoaderTarget = dispatcher.dispatchRemoteMethod("getResourceLoader");
      return enabled(new ResourceLoaderWrapper(dispatcher
          .newReflectionInvoker(dispatcher.invokeGetOnFeature(resourceLoaderTarget)),
                                               serializer));
    }
    return disabled();
  }

  @Override
  public ComponentLocationService componentLocationService() {
    Object componentLocationService = dispatcher.dispatchRemoteMethod("componentLocationService");
    return new ComponentLocationServiceWrapper(dispatcher.newReflectionInvoker(componentLocationService), serializer);
  }

  @Override
  public ArtifactAst getArtifactAst() {
    return (ArtifactAst) serializer.deserialize((String) dispatcher.dispatchRemoteMethod("getArtifactAst"));
  }

  @Override
  public void dispose() {
    dispatcher.dispatchRemoteMethod("dispose");
  }

}
