/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.bootstrap.internal;

import static com.google.common.collect.ImmutableSet.of;
import static com.google.common.io.Files.createTempDir;
import static java.lang.Integer.max;
import static java.lang.Runtime.getRuntime;
import static java.lang.String.format;
import static java.lang.System.clearProperty;
import static java.lang.System.identityHashCode;
import static java.lang.System.setProperty;
import static java.util.Objects.requireNonNull;
import static java.util.concurrent.Executors.newFixedThreadPool;
import static org.apache.commons.io.FileUtils.copyInputStreamToFile;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.commons.io.IOCase.SYSTEM;
import static org.apache.commons.lang3.ArrayUtils.addAll;
import static org.mule.runtime.module.embedded.internal.classloading.JdkOnlyClassLoaderFactory.create;

import org.mule.runtime.module.embedded.internal.classloading.FilteringClassLoader;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.ToolingRuntimeClientBuilderFactory;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrap;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;
import org.mule.tooling.client.bootstrap.internal.classloader.ToolingClassLoader;
import org.mule.tooling.client.bootstrap.internal.wrapper.ToolingRuntimeClientBuilderFactoryWrapper;
import org.mule.tooling.client.bootstrap.internal.wrapper.ToolingRuntimeClientBuilderFactoryWrapperBuilder;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Arrays;
import java.util.Optional;
import java.util.concurrent.ExecutorService;

import org.apache.commons.io.filefilter.SuffixFileFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Bootstrap for {@link ToolingRuntimeClient} that uses a folder containing jar files to determine the classpath for the
 * tooling-client.
 *
 * @since 1.4
 */
public class LibFolderToolingRuntimeClientBootstrap implements ToolingRuntimeClientBootstrap {

  public static final String CONFIG_DIR_NAME = "config";
  private static final int AVAILABLE_PROCESSORS = getRuntime().availableProcessors();

  private static final String LOG4J2_DISABLE_JMX = "log4j2.disable.jmx";
  private static final String LOG4J2_LOGGER_CONTEXT_FACTORY = "log4j2.loggerContextFactory";
  private static final String LOG4J2_IS_WEBAPP = "log4j2.is.webapp";
  public static final String TOOLING_LOG4J_CONTEXT_FACTORY_CLASSNAME =
      "org.mule.tooling.client.internal.log4j.ToolingLog4jContextFactory";

  private static Logger LOGGER = LoggerFactory.getLogger(LibFolderToolingRuntimeClientBootstrap.class);

  private final ToolingRuntimeClientBootstrapConfiguration configuration;
  private URLClassLoader toolingClassLoader;
  private ToolingRuntimeClientBuilderFactoryWrapper toolingRuntimeClientBuilderFactory;

  private File workingFolder;

  private ExecutorService executorService;
  private int threadNumber = 1;

  public LibFolderToolingRuntimeClientBootstrap(ToolingRuntimeClientBootstrapConfiguration configuration) {
    this(configuration, ToolingRuntimeClientBuilderFactoryWrapper.builder());
  }

  protected LibFolderToolingRuntimeClientBootstrap(ToolingRuntimeClientBootstrapConfiguration configuration,
                                                   ToolingRuntimeClientBuilderFactoryWrapperBuilder toolingRuntimeClientBuilderFactoryWrapperBuilder) {
    requireNonNull(configuration, "configuration cannot be null");
    this.configuration = configuration;

    LOGGER.info("Bootstrapping a Tooling Runtime Client version: {}", configuration.toolingVersion());

    // Log4j configuration for Tooling
    String currentLog4jDisableJmx = setProperty(LOG4J2_DISABLE_JMX, "true");
    // Avoiding Log4j thread locals. See https://logging.apache.org/log4j/2.x/manual/garbagefree.html
    String currentLog4jIsWebApp = setProperty(LOG4J2_IS_WEBAPP, "true");
    String currentLog4jLoggerContextFactory =
        setProperty(LOG4J2_LOGGER_CONTEXT_FACTORY, TOOLING_LOG4J_CONTEXT_FACTORY_CLASSNAME);
    try {
      workingFolder = configuration.workingFolder();
      if (workingFolder == null) {
        workingFolder = createTempDir();
      } else {
        if (!workingFolder.exists()) {
          if (!workingFolder.mkdirs()) {
            throw new RuntimeException(format("Could not create working directory: %s", workingFolder.getAbsolutePath()));
          }
        }
      }

      Optional<File> configDir = configuration.log4jConfiguration().map(log4JConfigurationFile -> {
        try (InputStream log4jConfigurationFileInputStream = log4JConfigurationFile.toURL().openStream()) {
          // Use log4j2-test.xml as it will be the first file to be loaded using TCCL
          File confDir = new File(workingFolder, CONFIG_DIR_NAME);
          confDir.deleteOnExit();
          copyInputStreamToFile(log4jConfigurationFileInputStream, new File(confDir, "log4j2-test.xml"));
          return confDir;
        } catch (Exception e) {
          throw new ToolingException("Error while writing log4j configuration file", e);
        }
      });

      this.toolingClassLoader = createClassLoader(configuration.toolingLibsFolder(), configuration.toolingVersion(), configDir);

      int nThreads = max(AVAILABLE_PROCESSORS, 2);
      if (configuration.executorServiceConfiguration().isPresent()) {
        nThreads = configuration.executorServiceConfiguration().get().maxNumberOfThreads();
      }
      executorService = newFixedThreadPool(nThreads,
                                           runnable -> {
                                             Thread thread = new Thread(runnable, format("ToolingClient-%s-%s", threadNumber++,
                                                                                         identityHashCode(this)));
                                             thread.setUncaughtExceptionHandler((t, e) -> LOGGER.error("Internal error", e));
                                             return thread;
                                           });

      this.toolingRuntimeClientBuilderFactory = toolingRuntimeClientBuilderFactoryWrapperBuilder
          .withToolingVersion(configuration.toolingVersion())
          .withToolingClassLoader(toolingClassLoader)
          .withExecutorService(executorService)
          .withMavenConfiguration(configuration.mavenConfiguration())
          .withWorkingDirectory(workingFolder)
          .build();
    } finally {
      resetSystemProperty(LOG4J2_DISABLE_JMX, currentLog4jDisableJmx);
      resetSystemProperty(LOG4J2_IS_WEBAPP, currentLog4jIsWebApp);
      resetSystemProperty(LOG4J2_LOGGER_CONTEXT_FACTORY, currentLog4jLoggerContextFactory);
    }
  }

  private static void resetSystemProperty(String propertyName, String oldValue) {
    if (oldValue == null) {
      clearProperty(propertyName);
    } else {
      setProperty(propertyName, oldValue);
    }
  }

  private URLClassLoader createClassLoader(File toolingLibsFolder, String toolingVersion, Optional<File> configDir) {
    try {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Creating URL class loader for Tooling Runtime Client version: {}", toolingVersion);
      }

      final FileFilter filter = new SuffixFileFilter(".jar", SYSTEM);
      URL[] urls = Arrays.stream(toolingLibsFolder.listFiles(filter))
          .map(jarFile -> {
            try {
              return jarFile.toURI().toURL();
            } catch (MalformedURLException e) {
              throw new ToolingException("Couldn't create the class loader for the Tooling Client", e);
            }
          })
          .toArray(size -> new URL[size]);

      FilteringClassLoader jdkOnlyClassLoader = create(this.getClass().getClassLoader(), of("io.takari.filemanager"));
      if (configDir.isPresent()) {
        urls = addAll(new URL[] {configDir.get().toURI().toURL()}, urls);
      }

      URLClassLoader classLoader =
          new ToolingClassLoader(configuration.muleVersion(), configuration.toolingVersion(), identityHashCode(this), urls,
                                 jdkOnlyClassLoader);

      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Created Tooling Client class loader: {}", classLoader);
      }

      return classLoader;
    } catch (Exception e) {
      throw new ToolingException("Couldn't create the class loader for the Tooling Client", e);
    }
  }

  @Override
  public ToolingRuntimeClientBuilderFactory getToolingRuntimeClientBuilderFactory() {
    if (toolingRuntimeClientBuilderFactory == null) {
      throw new IllegalStateException("Cannot be created a ToolingRuntimeClientBuilder once the bootstrap has been disposed");
    }
    return toolingRuntimeClientBuilderFactory;
  }

  @Override
  public String getMuleVersion() {
    return configuration.muleVersion();
  }

  @Override
  public String getToolingVersion() {
    return configuration.toolingVersion();
  }

  @Override
  public void dispose() {
    try {
      toolingRuntimeClientBuilderFactory.dispose();
    } finally {
      toolingRuntimeClientBuilderFactory = null;
    }

    executorService.shutdownNow();
    executorService = null;

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Closing Tooling Client class loader: {}", toolingClassLoader);
    }
    closeClassLoader(toolingClassLoader);
    toolingClassLoader = null;
    if (workingFolder != null) {
      deleteQuietly(workingFolder);
    }
  }

  private void closeClassLoader(URLClassLoader classLoader) {
    try {
      classLoader.close();
    } catch (IOException e) {
      // Do nothing
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Error while closing Tooling Client class loader", e);
      }
    }
  }

}
