/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal.wrapper;

import static java.util.Objects.requireNonNull;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.tooling.client.api.artifact.dsl.DslElementSyntax;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolver;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolverFactory;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolverService;
import org.mule.tooling.client.api.artifact.dsl.request.DslSyntaxResolverFactoryRequest;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.config.ConfigurationModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.construct.ConstructModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;
import org.mule.tooling.client.internal.serialization.Serializer;

import com.google.common.collect.ImmutableList;

import java.util.Optional;
import java.util.Set;

/**
 * Client side implementation for {@link DslSyntaxResolverService} that uses reflection and works as a bridge between
 * {@link ClassLoader classLoaders}.
 *
 * @since 1.0
 */
public class DslSyntaxResolverServiceWrapper implements DslSyntaxResolverService {

  private Dispatcher dispatcher;
  private Serializer serializer;

  public DslSyntaxResolverServiceWrapper(Dispatcher dispatcher, Serializer serializer) {
    requireNonNull(dispatcher, "dispatcher cannot be null");
    requireNonNull(serializer, "serializer cannot be null");

    this.dispatcher = dispatcher;
    this.serializer = serializer;
  }

  @Override
  public DslSyntaxResolverFactory getDslSyntaxResolverFactory(DslSyntaxResolverFactoryRequest request) {
    Object dslSyntaxResolverFactoryTarget = dispatcher.dispatchRemoteMethod("getDslSyntaxResolverFactory",
                                                                            ImmutableList
                                                                                .of(DslSyntaxResolverFactoryRequest.class),
                                                                            ImmutableList.of(serializer.serialize(request)));
    return new DslSyntaxResolverFactoryWrapper(dispatcher.newReflectionInvoker(dslSyntaxResolverFactoryTarget));
  }

  private class DslSyntaxResolverFactoryWrapper implements DslSyntaxResolverFactory {

    private Dispatcher dispatcher;

    public DslSyntaxResolverFactoryWrapper(Dispatcher dispatcher) {
      this.dispatcher = dispatcher;
    }

    @Override
    public void dispose() {
      dispatcher.dispatchRemoteMethod("dispose");
    }

    @Override
    public DslSyntaxResolver createDslResolver(ExtensionModel extensionModel) {
      Object dslSyntaxResolverTarget = dispatcher.dispatchRemoteMethod("createDslResolver",
                                                                       ImmutableList.of(ExtensionModel.class),
                                                                       ImmutableList.of(serializer.serialize(extensionModel)));
      return new DslSyntaxResolverWrapper(dispatcher.newReflectionInvoker(dslSyntaxResolverTarget));
    }

    private class DslSyntaxResolverWrapper implements DslSyntaxResolver {

      private Dispatcher dispatcher;

      public DslSyntaxResolverWrapper(Dispatcher dispatcher) {
        this.dispatcher = dispatcher;
      }

      @Override
      public void dispose() {
        dispatcher.dispatchRemoteMethod("dispose");
      }

      @Override
      public DslElementSyntax resolve(ConfigurationModel component) {
        return dispatchRemoteMethod("resolve", component, ConfigurationModel.class);
      }

      @Override
      public DslElementSyntax resolve(ConnectionProviderModel component) {
        return dispatchRemoteMethod("resolve", component, ConnectionProviderModel.class);
      }

      @Override
      public DslElementSyntax resolve(ConstructModel component) {
        return dispatchRemoteMethod("resolve", component, ConstructModel.class);
      }

      @Override
      public DslElementSyntax resolve(OperationModel component) {
        return dispatchRemoteMethod("resolve", component, OperationModel.class);
      }

      @Override
      public DslElementSyntax resolve(SourceModel component) {
        return dispatchRemoteMethod("resolve", component, SourceModel.class);
      }

      @Override
      public Optional<DslElementSyntax> resolve(MetadataType type) {
        return dispatchRemoteMethodOptionalReturn("resolve", type, MetadataType.class);
      }

      @Override
      public Set<ObjectType> getSubTypes(ObjectType type) {
        return dispatchRemoteMethod("getSubTypes", type, ObjectType.class);
      }

      private <T> T dispatchRemoteMethod(String method, Object parameter, Class parameterType) {
        return serializer.deserialize(doDispatch(method, parameter, parameterType));
      }

      private Optional<DslElementSyntax> dispatchRemoteMethodOptionalReturn(String method, Object parameter,
                                                                            Class parameterType) {
        return (Optional<DslElementSyntax>) serializer.deserialize(doDispatch(method, parameter, parameterType));
      }

      private String doDispatch(String method, Object parameter, Class parameterType) {
        return (String) dispatcher.dispatchRemoteMethod(method,
                                                        ImmutableList.of(parameterType),
                                                        ImmutableList.of(serializer.serialize(parameter)));
      }

    }

  }

}
