/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.api.sampledata;

import static java.util.Optional.ofNullable;

import org.mule.tooling.event.model.MessageModel;

import java.util.Optional;

/**
 * Represents the result of a sample data connector request using {@link MessageModel}
 *
 * @since 4.4.0
 */
public class SampleDataMessageModelResult {

  private SampleDataMessageModelFailure failure;
  private MessageModel messageModel;

  /**
   * Creates a new {@link SampleDataMessageModelResult} with a successful result containing the resolved sample data.
   *
   * @param messageModel The {@MessageModel} with the sample data
   */
  public SampleDataMessageModelResult(MessageModel messageModel) {
    this.messageModel = messageModel;
  }

  /**
   * Creates a new {@link SampleDataMessageModelResult} with a failure result containing the associated
   * {@link SampleDataMessageModelFailure failure}
   *
   * @param failure The generated failure occurred trying to resolve the sample data
   */
  public SampleDataMessageModelResult(SampleDataMessageModelFailure failure) {
    this.failure = failure;
  }

  /**
   * @return An {@link Optional} {@link MessageModel} with the sample data. In an error case returns empty.
   */
  public Optional<MessageModel> getSampleData() {
    return ofNullable(messageModel);
  }

  /**
   * @return An {@link Optional} {@link SampleDataMessageModelFailure failure} with the error that occurred trying to resolve the
   *         sample data
   */
  public Optional<SampleDataMessageModelFailure> getFailure() {
    return ofNullable(failure);
  }

  /**
   * @return A boolean indicating if the resolution finished correctly or not.
   */
  public boolean isSuccess() {
    return !getFailure().isPresent();
  }

}
