/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.internal.handler.session;

import static com.mulesoft.agent.domain.tooling.TestConnectionStatus.testConnectionResult;
import static java.lang.String.format;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static javax.ws.rs.core.MediaType.TEXT_PLAIN;
import static javax.ws.rs.core.Response.Status.BAD_REQUEST;
import static javax.ws.rs.core.Response.Status.INTERNAL_SERVER_ERROR;
import static javax.ws.rs.core.Response.Status.NOT_FOUND;
import static org.mule.tooling.api.serialization.SerializerFactory.jsonSerializer;
import static org.mule.tooling.internal.sampledata.SampleDataResultTransformer.toComponentSampleDataResult;

import org.mule.runtime.app.declaration.api.ComponentElementDeclaration;
import org.mule.runtime.extension.api.persistence.metadata.ComponentMetadataTypesDescriptorResultJsonSerializer;
import org.mule.runtime.extension.api.persistence.metadata.MetadataKeysResultJsonSerializer;
import org.mule.runtime.extension.api.persistence.value.ValueResultJsonSerializer;
import org.mule.runtime.module.repository.api.BundleNotFoundException;
import org.mule.runtime.module.tooling.api.ArtifactAgnosticServiceBuilder;
import org.mule.tooling.api.request.session.DeclarationSessionCreationRequest;
import org.mule.tooling.api.request.session.model.Dependency;
import org.mule.tooling.api.request.values.FieldValuesRequest;
import org.mule.tooling.api.request.values.ValuesRequest;
import org.mule.tooling.api.serialization.Serializer;
import org.mule.tooling.api.service.Configuration;
import org.mule.tooling.api.service.expression.RuntimeAgentService;
import org.mule.tooling.api.service.session.DeclarationSessionContainer;
import org.mule.tooling.api.service.session.DeclarationSessionAgentService;
import org.mule.tooling.internal.handler.LoggingHandler;

import com.mulesoft.agent.exception.AgentEnableOperationException;
import com.mulesoft.agent.handlers.ExternalMessageHandler;
import com.mulesoft.agent.util.ResponseHelper;

import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import com.google.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.HeaderParam;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Response;

@Singleton
@Named("org.mule.tooling.handler")
@Path("/tooling")
public class DeclarationSessionsAgentHandler implements ExternalMessageHandler, LoggingHandler {

  private static final String IGNORE_CACHE_QUERY_PARAM = "ignoreCache";
  private static final String VERBOSE_QUERY_PARAM = "verbose";

  private static final String BASE_SESSIONS_PATH = "/declaration-sessions";
  private Serializer serializer = jsonSerializer();

  @Inject
  private DeclarationSessionAgentService declarationSessionService;

  @Inject
  private RuntimeAgentService runtimeAgentService;

  @Inject
  private Configuration configuration;

  private AtomicBoolean enabled = new AtomicBoolean();
  private ResponseHelper responseHelper = new ResponseHelper()
      .addErrorMapping(BundleNotFoundException.class, BAD_REQUEST);

  public void enable(boolean state) throws AgentEnableOperationException {
    enabled.set(state);
  }

  public boolean isEnabled() {
    return enabled.get();
  }

  @POST
  @Path(BASE_SESSIONS_PATH)
  @Consumes(APPLICATION_JSON)
  @Produces(TEXT_PLAIN)
  public Response createSession(@HeaderParam(REQUEST_ID_HEADER_NAME) String requestIdHeader,
                                String declarationSessionCreationRequestString,
                                @QueryParam(VERBOSE_QUERY_PARAM) boolean verboseErrors) {
    return withLoggingContext(ofNullable(requestIdHeader), () -> {
      try {
        DeclarationSessionCreationRequest sessionCreationRequest =
            serializer.deserialize(declarationSessionCreationRequestString, DeclarationSessionCreationRequest.class);
        DeclarationSessionContainer configService =
            declarationSessionService.createDeclarationSession(toMuleDependencies(sessionCreationRequest.getDependencies()),
                                                               sessionCreationRequest.getArtifactDeclaration(),
                                                               sessionCreationRequest.getSessionProperties());
        return responseHelper.response(configService::getId, verboseErrors);
      } catch (Exception e) {
        return responseHelper.failure(INTERNAL_SERVER_ERROR, e.getMessage(), e, verboseErrors);
      }
    });
  }

  private List<ArtifactAgnosticServiceBuilder.Dependency> toMuleDependencies(List<Dependency> dependencies) {
    return dependencies.stream().map(d -> {
      ArtifactAgnosticServiceBuilder.Dependency dependency = new ArtifactAgnosticServiceBuilder.Dependency();
      dependency.setArtifactId(d.getArtifactId());
      dependency.setClassifier(d.getClassifier());
      dependency.setExclusions(d.getExclusions().stream().map(e -> {
        ArtifactAgnosticServiceBuilder.Exclusion exclusion = new ArtifactAgnosticServiceBuilder.Exclusion();
        exclusion.setArtifactId(e.getArtifactId());
        exclusion.setGroupId(e.getGroupId());
        return exclusion;
      }).collect(toList()));
      dependency.setGroupId(d.getGroupId());
      dependency.setOptional(d.getOptional());
      dependency.setScope(d.getScope());
      dependency.setSystemPath(d.getSystemPath());
      dependency.setType(d.getType());
      dependency.setVersion(d.getVersion());
      return dependency;
    }).collect(toList());
  }

  @DELETE
  @Path(BASE_SESSIONS_PATH + "/{sessionId}")
  public boolean deleteSession(@HeaderParam(REQUEST_ID_HEADER_NAME) String requestIdHeader,
                               @PathParam("sessionId") String sessionId) {
    return withLoggingContext(ofNullable(requestIdHeader),
                              () -> declarationSessionService.deleteSession(sessionId));
  }

  @GET
  @Path(BASE_SESSIONS_PATH + "/{sessionId}/connection")
  @Produces(APPLICATION_JSON)
  public Response testConnection(@HeaderParam(REQUEST_ID_HEADER_NAME) String requestIdHeader,
                                 @PathParam("sessionId") String sessionId,
                                 @QueryParam("configName") String configName,
                                 @QueryParam(VERBOSE_QUERY_PARAM) boolean verboseErrors) {
    return withLoggingContext(ofNullable(requestIdHeader), () -> declarationSessionService.getSession(sessionId)
        .map(sc -> responseHelper.response(() -> serializer
            .serialize(testConnectionResult(sc.getSession().testConnection(configName), true)), verboseErrors))
        .orElseGet(() -> sessionNotFoundResponse(sessionId)));
  }

  private Response sessionNotFoundResponse(String sessionId) {
    return responseHelper.failure(NOT_FOUND, format("No session found for id: '%s'", sessionId),
                                  new IllegalArgumentException("No session found"),
                                  false);
  }

  @PUT
  @Path(BASE_SESSIONS_PATH + "/{sessionId}/values")
  @Consumes(APPLICATION_JSON)
  @Produces(APPLICATION_JSON)
  public Response getValues(@HeaderParam(REQUEST_ID_HEADER_NAME) String requestIdHeader,
                            @PathParam("sessionId") String sessionId,
                            String valuesRequestString, @QueryParam(VERBOSE_QUERY_PARAM) boolean verboseErrors) {
    return withLoggingContext(ofNullable(requestIdHeader), () -> {
      ValuesRequest valuesRequest;
      try {
        valuesRequest = serializer.deserialize(valuesRequestString, ValuesRequest.class);
      } catch (Exception e) {
        return responseHelper.failure(BAD_REQUEST, "Body is not valid json", e, verboseErrors);
      }
      if (!valuesRequest.getParameterizedElementDeclaration().isPresent()) {
        return responseHelper.failure(BAD_REQUEST, "No parameterized element declaration found in request",
                                      new IllegalStateException("Invalid ValuesRequest"), verboseErrors);
      }
      return declarationSessionService
          .getSession(sessionId)
          .map(sc -> responseHelper.response(() -> new ValueResultJsonSerializer().serialize(sc.getSession().getValues(
                                                                                                                       valuesRequest
                                                                                                                           .getParameterizedElementDeclaration()
                                                                                                                           .get(),
                                                                                                                       valuesRequest
                                                                                                                           .getProviderName())),
                                             verboseErrors))
          .orElseGet(() -> sessionNotFoundResponse(sessionId));
    });
  }

  @PUT
  @Path(BASE_SESSIONS_PATH + "/{sessionId}/fieldValues")
  @Consumes(APPLICATION_JSON)
  @Produces(APPLICATION_JSON)
  public Response getFieldValues(@HeaderParam(REQUEST_ID_HEADER_NAME) String requestIdHeader,
                                 @PathParam("sessionId") String sessionId,
                                 String valuesRequestString,
                                 @QueryParam("verbose") boolean verboseErrors) {
    return withLoggingContext(ofNullable(requestIdHeader), () -> {
      FieldValuesRequest fieldValuesRequest;
      try {
        fieldValuesRequest = serializer.deserialize(valuesRequestString, FieldValuesRequest.class);
      } catch (Exception e) {
        return responseHelper.failure(BAD_REQUEST, "Body is not valid json", e, verboseErrors);
      }
      if (fieldValuesRequest.getComponentElementDeclaration() == null) {
        return responseHelper.failure(BAD_REQUEST, "No component element declaration found in request",
                                      new IllegalStateException("Invalid FieldValuesRequest"), verboseErrors);
      }
      return declarationSessionService
          .getSession(sessionId)
          .map(sc -> responseHelper.response(() -> new ValueResultJsonSerializer().serialize(
                                                                                             sc.getSession().getFieldValues(
                                                                                                                            fieldValuesRequest
                                                                                                                                .getComponentElementDeclaration(),
                                                                                                                            fieldValuesRequest
                                                                                                                                .getProviderName(),
                                                                                                                            fieldValuesRequest
                                                                                                                                .getTargetSelector())),
                                             verboseErrors))
          .orElseGet(() -> sessionNotFoundResponse(sessionId));
    });
  }

  @PUT
  @Path(BASE_SESSIONS_PATH + "/{sessionId}/component/metadata")
  @Consumes(APPLICATION_JSON)
  @Produces(APPLICATION_JSON)
  public Response componentMetadata(@HeaderParam(REQUEST_ID_HEADER_NAME) String requestIdHeader,
                                    @PathParam("sessionId") String sessionId,
                                    String componentElementDeclarationString,
                                    @QueryParam(IGNORE_CACHE_QUERY_PARAM) boolean ignoreCache,
                                    @QueryParam(VERBOSE_QUERY_PARAM) boolean verboseErrors) {
    return withLoggingContext(ofNullable(requestIdHeader), () -> {
      ComponentElementDeclaration componentElementDeclaration;
      try {
        componentElementDeclaration =
            serializer.deserialize(componentElementDeclarationString, ComponentElementDeclaration.class);
      } catch (Exception e) {
        return responseHelper.failure(BAD_REQUEST, "Body is not a valid json", e, verboseErrors);
      }
      return declarationSessionService
          .getSession(sessionId)
          .map(sc -> {
            if (ignoreCache) {
              sc.getSession().disposeMetadataCache(componentElementDeclaration);
            }
            return responseHelper.response(() -> getComponentMetadataTypesDescriptorResultJsonSerializer()
                .serialize(sc.getSession().resolveComponentMetadata(componentElementDeclaration)), verboseErrors);
          })
          .orElseGet(() -> sessionNotFoundResponse(sessionId));
    });
  }

  @PUT
  @Path(BASE_SESSIONS_PATH + "/{sessionId}/component/metadata/keys")
  @Consumes(APPLICATION_JSON)
  @Produces(APPLICATION_JSON)
  public Response metadataKeys(@HeaderParam(REQUEST_ID_HEADER_NAME) String requestIdHeader,
                               @PathParam("sessionId") String sessionId,
                               String componentElementDeclarationString,
                               @QueryParam(IGNORE_CACHE_QUERY_PARAM) boolean ignoreCache,
                               @QueryParam(VERBOSE_QUERY_PARAM) boolean verboseErrors) {
    return withLoggingContext(ofNullable(requestIdHeader), () -> {
      ComponentElementDeclaration componentElementDeclaration;
      try {
        componentElementDeclaration =
            serializer.deserialize(componentElementDeclarationString, ComponentElementDeclaration.class);
      } catch (Exception e) {
        return responseHelper.failure(BAD_REQUEST, "Body is not a valid json", e, verboseErrors);
      }
      return declarationSessionService
          .getSession(sessionId)
          .map(sc -> {
            if (ignoreCache) {
              sc.getSession().disposeMetadataCache(componentElementDeclaration);
            }
            return responseHelper.response(() -> new MetadataKeysResultJsonSerializer()
                .serialize(sc.getSession().getMetadataKeys(componentElementDeclaration)), verboseErrors);
          })
          .orElseGet(() -> sessionNotFoundResponse(sessionId));
    });
  }

  @PUT
  @Path(BASE_SESSIONS_PATH + "/{sessionId}/component/sampledata")
  @Consumes(APPLICATION_JSON)
  @Produces(APPLICATION_JSON)
  public Response getSampleData(@HeaderParam(REQUEST_ID_HEADER_NAME) String requestIdHeader,
                                @PathParam("sessionId") String sessionId,
                                String componentElementDeclarationString,
                                @QueryParam(VERBOSE_QUERY_PARAM) boolean verboseErrors) {
    return withLoggingContext(ofNullable(requestIdHeader), () -> {
      ComponentElementDeclaration componentElementDeclaration;
      try {
        componentElementDeclaration =
            serializer.deserialize(componentElementDeclarationString, ComponentElementDeclaration.class);
      } catch (Exception e) {
        return responseHelper.failure(BAD_REQUEST, "Body is not a valid json", e, verboseErrors);
      }
      return declarationSessionService
          .getSession(sessionId)
          .map(sc -> responseHelper.response(() -> serializer
              .serialize(toComponentSampleDataResult(sc.getSession().getSampleData(componentElementDeclaration),
                                                     runtimeAgentService.getExpressionLanguage(),
                                                     configuration.getSampleDataMaxPayloadSize())),
                                             verboseErrors))
          .orElseGet(() -> sessionNotFoundResponse(sessionId));
    });
  }

  // TODO MULE-10794 : Cannot reuse serializer between calls
  private ComponentMetadataTypesDescriptorResultJsonSerializer getComponentMetadataTypesDescriptorResultJsonSerializer() {
    return new ComponentMetadataTypesDescriptorResultJsonSerializer(false, true);
  }

}
