/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.internal.handler.sampledata;

import static java.util.Optional.ofNullable;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static javax.ws.rs.core.Response.Status.INTERNAL_SERVER_ERROR;
import static javax.ws.rs.core.Response.Status.NOT_FOUND;
import static org.mule.runtime.api.component.location.Location.builderFromStringRepresentation;
import static org.mule.tooling.api.serialization.SerializerFactory.jsonSerializer;
import static org.mule.tooling.internal.sampledata.SampleDataResultTransformer.toComponentSampleDataResult;

import org.mule.tooling.api.service.Configuration;
import org.mule.tooling.api.service.expression.RuntimeAgentService;
import org.mule.tooling.api.service.artifact.DeployableArtifactAgentService;
import org.mule.tooling.api.serialization.Serializer;
import org.mule.tooling.internal.handler.LoggingHandler;

import com.mulesoft.agent.exception.AgentEnableOperationException;
import com.mulesoft.agent.exception.NoSuchApplicationException;
import com.mulesoft.agent.handlers.ExternalMessageHandler;
import com.mulesoft.agent.services.ToolingService;
import com.mulesoft.agent.util.ResponseHelper;

import java.util.NoSuchElementException;
import java.util.concurrent.atomic.AtomicBoolean;

import com.google.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.HeaderParam;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Response;

@Singleton
@Named("org.mule.tooling.sampledata.handler")
@Path("/tooling")
public class SampleDataServiceAgentHandler implements ExternalMessageHandler, LoggingHandler {

  private static final String VERBOSE_QUERY_PARAM = "verbose";

  private static final String APPLICATIONS = "applications";
  private static final String COMPONENTS = "components";
  private static final String APP_ID = "applicationId";
  private static final String COMPONENT_PATH = "componentPath";
  private static final String SAMPLE_DATA = "sampledata";

  private final AtomicBoolean enabled = new AtomicBoolean();
  private final ResponseHelper responseHelper = new ResponseHelper()
      .addErrorMapping(NoSuchApplicationException.class, NOT_FOUND)
      .addErrorMapping(NoSuchElementException.class, NOT_FOUND);
  private final Serializer serializer = jsonSerializer();

  @Inject
  private ToolingService toolingService;

  @Inject
  private RuntimeAgentService runtimeAgentService;

  @Inject
  private DeployableArtifactAgentService deployableArtifactService;

  @Inject
  private Configuration configuration;

  @Override
  public void enable(boolean b) throws AgentEnableOperationException {
    enabled.set(true);
  }

  @Override
  public boolean isEnabled() {
    return enabled.get();
  }

  @GET
  @Path(APPLICATIONS +
      "/{" + APP_ID + "}/" +
      COMPONENTS +
      "/{" + COMPONENT_PATH + "}/" +
      SAMPLE_DATA)
  @Consumes(APPLICATION_JSON)
  @Produces(APPLICATION_JSON)
  public Response getSampleData(@HeaderParam(REQUEST_ID_HEADER_NAME) String requestIdHeader,
                                @PathParam(APP_ID) String appId,
                                @PathParam(COMPONENT_PATH) String componentPath,
                                @QueryParam(VERBOSE_QUERY_PARAM) boolean verboseErrors) {
    try {
      return responseHelper.response(
                                     () -> withLoggingContext(ofNullable(requestIdHeader),
                                                              () -> serializer.serialize(
                                                                                         toComponentSampleDataResult(
                                                                                                                     deployableArtifactService
                                                                                                                         .getSampleData(
                                                                                                                                        toolingService
                                                                                                                                            .getDeployedApplication(appId),
                                                                                                                                        builderFromStringRepresentation(componentPath)
                                                                                                                                            .build()),
                                                                                                                     runtimeAgentService
                                                                                                                         .getExpressionLanguage(),
                                                                                                                     configuration
                                                                                                                         .getSampleDataMaxPayloadSize()))));
    } catch (Exception e) {
      return responseHelper.failure(INTERNAL_SERVER_ERROR, e.getMessage(), e, verboseErrors);
    }
  }
}
