/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.internal.service.session;

import static java.lang.String.format;
import static java.util.Optional.ofNullable;

import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.runtime.core.api.util.UUID;
import org.mule.runtime.module.tooling.api.ArtifactAgnosticServiceBuilder;
import org.mule.runtime.module.tooling.api.ToolingService;
import org.mule.runtime.module.tooling.api.ToolingServiceAware;
import org.mule.runtime.module.tooling.api.artifact.DeclarationSessionBuilder;
import org.mule.tooling.api.service.session.DeclarationSessionContainer;
import org.mule.tooling.api.service.session.DeclarationSessionAgentService;
import org.mule.tooling.internal.service.AbstractConfigurableService;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

import javax.inject.Named;
import javax.inject.Singleton;

@Named("org.mule.tooling.service.session")
@Singleton
public class DefaultDeclarationSessionAgentService extends AbstractConfigurableService
    implements DeclarationSessionAgentService, ToolingServiceAware {

  private Map<String, DeclarationSessionContainer> createdSessions = new ConcurrentHashMap<>();
  private ToolingService muleToolingService;

  @Override
  public void setToolingService(ToolingService toolingService) {
    this.muleToolingService = toolingService;
  }

  @Override
  public DeclarationSessionContainer createDeclarationSession(List<ArtifactAgnosticServiceBuilder.Dependency> dependencies,
                                                              ArtifactDeclaration artifactDeclaration,
                                                              Map<String, String> sessionProperties) {
    String id = UUID.getUUID();
    if (this.createdSessions.containsKey(id)) {
      throw new IllegalStateException(format("A session with id: %s is already created ", id));
    }
    final DeclarationSessionBuilder builder = this.muleToolingService.newDeclarationSessionBuilder();
    dependencies.forEach(builder::addDependency);
    builder.setArtifactDeclaration(artifactDeclaration);
    builder.setArtifactProperties(sessionProperties);
    final DeclarationSessionContainer declarationSession = new DefaultDeclarationSessionContainer(id, builder.build());
    this.createdSessions.put(id, declarationSession);
    return declarationSession;
  }

  @Override
  public Optional<DeclarationSessionContainer> getSession(String id) {
    return ofNullable(createdSessions.get(id));
  }

  @Override
  public boolean deleteSession(String id) {
    DeclarationSessionContainer sessionContainer = createdSessions.remove(id);
    if (sessionContainer != null) {
      sessionContainer.getSession().dispose();
      return true;
    }
    return false;
  }

}
