/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.internal.sampledata;

import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static org.mule.runtime.api.message.Message.builder;
import static org.mule.runtime.api.message.Message.of;
import static org.mule.runtime.api.sampledata.SampleDataResult.resultFrom;
import static org.mule.tooling.internal.sampledata.SampleDataResultTransformer.toComponentSampleDataResult;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.streaming.bytes.CursorStream;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.core.internal.streaming.bytes.ByteArrayCursorStreamProvider;
import org.mule.tck.junit4.AbstractMuleContextTestCase;
import org.mule.tooling.api.sampledata.SampleDataMessageModelResult;
import org.mule.tooling.event.model.MessageModel;

import java.io.ByteArrayInputStream;
import java.io.InputStream;

import org.junit.Test;

public class SampleDataResultTransformerTestCase extends AbstractMuleContextTestCase {

  @Test
  public void stringTypedValue() {
    String content = "Hello World";
    assertSampleDataMessageModelResult(content, content, "*/*; charset=UTF-8", "java.lang.String");
  }

  @Test
  public void pojoTypedValue() {
    Person person = new Person("rene", 71);
    String output = "{\n  name: \"rene\",\n  age: 71\n}";
    assertSampleDataMessageModelResult(person, output, "application/dw; charset=UTF-8", "java.lang.String");
  }

  @Test
  public void byteArrayTypedValue() {
    String output = "Hello World";
    byte[] content = output.getBytes();
    assertSampleDataMessageModelResult(content, output, "*/*", "[B");
  }

  @Test
  public void inputStreamTypedValue() {
    String output = "Hello World";
    InputStream content = new ByteArrayInputStream(output.getBytes());
    assertSampleDataMessageModelResult(content, output, "*/*; charset=UTF-8", "java.io.ByteArrayInputStream");
  }

  @Test
  public void cursorStreamProviderTypedValue() {
    String output = "Hello World";
    CursorStreamProvider content = new ByteArrayCursorStreamProvider(output.getBytes());
    assertSampleDataMessageModelResult(content, output, "*/*",
                                       "org.mule.runtime.core.internal.streaming.bytes.ByteArrayCursorStreamProvider");
  }

  @Test
  public void forceSerializationException() {
    CursorStreamProvider content = new ExpectedExceptionCursorStreamProvider();

    SampleDataMessageModelResult sampleDataMessageModelResult = toComponentSampleDataResult(resultFrom(of(content)), el(), 9);
    assertThat(sampleDataMessageModelResult.isSuccess(), is(false));
    assertThat(sampleDataMessageModelResult.getFailure().isPresent(), is(true));
    assertThat(sampleDataMessageModelResult.getFailure().get().getMessage(), is("Sample Data serialization error"));
    assertThat(sampleDataMessageModelResult.getFailure().get().getReason(),
               is("java.lang.IllegalStateException: Expected exception"));
    assertThat(sampleDataMessageModelResult.getFailure().get().getFailureCode(), is("SERIALIZATION_ERROR"));
  }

  @Test
  public void payloadSizeExceeded() {
    String content = "1234567890";
    assertPayloadSizeExceeded(of(content));
  }

  @Test
  public void attributesSizeExceeded() {
    String content = "1234567890";
    Message message = builder(of(null)).attributesValue(content).build();
    assertPayloadSizeExceeded(message);
  }

  private void assertPayloadSizeExceeded(Message message) {
    SampleDataMessageModelResult sampleDataMessageModelResult = toComponentSampleDataResult(resultFrom(message), el(), 9);
    assertThat(sampleDataMessageModelResult.isSuccess(), is(false));
    assertThat(sampleDataMessageModelResult.getFailure().isPresent(), is(true));
    assertThat(sampleDataMessageModelResult.getFailure().get().getMessage(), is("Content size is too large"));
    assertThat(sampleDataMessageModelResult.getFailure().get().getReason(),
               is("Content size of 10 bytes exceeds allowed maximum of 9 bytes"));
    assertThat(sampleDataMessageModelResult.getFailure().get().getFailureCode(), is("CONTENT_LIMIT_ERROR"));
  }

  private void assertSampleDataMessageModelResult(Object payload, String output, String mediaType, String type) {
    SampleDataMessageModelResult sampleDataMessageModelResult = toComponentSampleDataResult(resultFrom(of(payload)), el(), 4096);
    assertThat(sampleDataMessageModelResult.isSuccess(), is(true));
    assertThat(sampleDataMessageModelResult.getSampleData().isPresent(), is(true));
    MessageModel messageModel = sampleDataMessageModelResult.getSampleData().get();
    assertThat(new String(messageModel.getPayload().getContent()), is(output));
    assertThat(messageModel.getPayload().getDataType().getMediaType(), is(mediaType));
    assertThat(messageModel.getPayload().getDataType().getType(), is(type));
  }

  private ExpressionLanguage el() {
    return muleContext.getExpressionManager();
  }

  private static class Person {

    private String name;
    private int age;

    public Person(String name, int age) {
      this.name = name;
      this.age = age;
    }

    public String getName() {
      return name;
    }

    public int getAge() {
      return age;
    }
  }

  private static class ExpectedExceptionCursorStreamProvider implements CursorStreamProvider {

    @Override
    public CursorStream openCursor() {
      throw new IllegalStateException("Expected exception");
    }

    @Override
    public void close() {}

    @Override
    public void releaseResources() {}

    @Override
    public boolean isClosed() {
      return false;
    }

  }
}
