/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.internal.utils;

import static junit.framework.Assert.assertNull;
import static org.apache.commons.lang3.builder.ToStringStyle.SHORT_PREFIX_STYLE;
import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyString;
import static org.mockito.Mockito.RETURNS_DEEP_STUBS;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.api.metadata.DataType.STRING;
import static org.mule.runtime.core.api.exception.Errors.CORE_NAMESPACE_NAME;
import static org.mule.runtime.core.api.exception.Errors.Identifiers.ANY_IDENTIFIER;
import static org.mule.tooling.internal.utils.MuleEventTransformer.getEventModel;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.message.Error;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.object.CursorIterator;
import org.mule.runtime.api.streaming.object.CursorIteratorProvider;
import org.mule.runtime.core.api.streaming.bytes.InMemoryCursorStreamConfig;
import org.mule.runtime.core.api.streaming.bytes.InMemoryCursorStreamProvider;
import org.mule.runtime.core.internal.message.ErrorBuilder;
import org.mule.runtime.core.internal.message.InternalMessage;
import org.mule.runtime.core.internal.streaming.bytes.PoolingByteBufferManager;
import org.mule.tooling.event.model.EventModel;
import org.mule.tooling.event.model.MessageModel;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;

import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.junit.Test;
import org.mockito.stubbing.Answer;

public class MuleEventTransformerTestCase {

  private static final String STRING_PAYLOAD = "payload";
  private static final String ERROR_PAYLOAD = "{\"error\": \"true\"}";
  private static final String ERROR_DESCRIPTION = "errorDescription";
  private static final String ERROR_DETAILED_DESCRIPTION = "errorDetailedDescription";
  private static final String PAYLOAD = "{\"payload\": \"aPayload\"}";
  private static final String BIG_PAYLOAD = "{\"payload\": \"aBigPayload\"}";
  private static final Charset UTF_16 = Charset.forName("UTF-16");
  private static final Charset UTF_8 = Charset.forName("UTF-8");
  private static final int CURSOR_ITERATOR_PAYLOADS = 3;

  private static final String PAYLOAD_BINDING_CONTEXT_KEY = "payload";

  @Test
  public void getEventModelAttributes() {
    final TestAttributes attributes = new TestAttributes("name", "fileName", 1l, new HashMap<>());
    Message muleMessage = getMockedMuleMessage(DataType.JSON_STRING, PAYLOAD, attributes);
    ExpressionLanguage expressionLanguage = getMockedExpressionLanguage();

    EventModel eventModel = getEventModel(getEvent(muleMessage), expressionLanguage, -1, -1, UTF_8);

    assertNotNull(eventModel.getMessage().getAttributes().getContent());
    assertEquals(attributes.toString(), new String(eventModel.getMessage().getAttributes().getContent(), UTF_8));
  }

  @Test
  public void getEventModelAttributesAsXML() {
    String XML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><example>hello</example>";
    TypedValue attributes = new TypedValue(XML, DataType.XML_STRING);

    Message muleMessage = getMockedMuleMessageWithTypedValue(DataType.XML_STRING, PAYLOAD, attributes);
    EventModel eventModel = getEventModel(getEvent(muleMessage), mock(ExpressionLanguage.class), -1, -1, UTF_8);

    assertNotNull(eventModel.getMessage().getAttributes().getContent());
    assertThat(new String(eventModel.getMessage().getAttributes().getContent()), is(XML));
    assertThat(eventModel.getMessage().getAttributes().getDataType().getMediaType(),
               is(MediaType.XML.withCharset(UTF_8).toRfcString()));
  }

  @Test
  public void getEventModelReturnsValueForMuleMessageDataType() {
    Message muleMessage =
        getMockedMuleMessage(DataType.MULE_MESSAGE,
                             InternalMessage.builder().value(PAYLOAD).mediaType(MediaType.APPLICATION_JSON).build());
    ExpressionLanguage expressionLanguage = getMockedExpressionLanguage(typedValue -> new TypedValue(
                                                                                                     ((Message) typedValue
                                                                                                         .getValue()).getPayload()
                                                                                                             .getValue(),
                                                                                                     STRING));

    EventModel eventModel = getEventModel(getEvent(muleMessage), expressionLanguage, -1, -1, UTF_8);

    assertEquals(PAYLOAD, new String(eventModel.getMessage().getPayload().getContent(), UTF_8));
  }

  @Test
  public void getEventModelWithError() {
    Message muleMessage =
        Message.builder().value(Message.builder().value(PAYLOAD).mediaType(MediaType.APPLICATION_JSON).build()).build();
    Message errorMuleMessage =
        getMockedMuleMessage(DataType.MULE_MESSAGE,
                             InternalMessage.builder().value(ERROR_PAYLOAD).mediaType(MediaType.APPLICATION_JSON).build());
    ExpressionLanguage expressionLanguage = getMockedExpressionLanguage(typedValue -> new TypedValue(
                                                                                                     ((Message) typedValue
                                                                                                         .getValue()).getPayload()
                                                                                                             .getValue(),
                                                                                                     STRING));

    ErrorType errorType = mock(ErrorType.class);
    when(errorType.getIdentifier()).thenReturn(ANY_IDENTIFIER);
    when(errorType.getNamespace()).thenReturn(CORE_NAMESPACE_NAME);

    Error muleError = ErrorBuilder.builder()
        .exception(new RuntimeException())
        .description(ERROR_DESCRIPTION)
        .detailedDescription(ERROR_DETAILED_DESCRIPTION)
        .errorType(errorType)
        .errorMessage(errorMuleMessage)
        .build();

    Event event = mock(Event.class, RETURNS_DEEP_STUBS);
    when(event.getMessage()).thenReturn(muleMessage);
    when(event.getError()).thenReturn(Optional.of(muleError));
    when(event.getVariables()).thenReturn(new HashMap<>());

    EventModel eventModel = getEventModel(event, expressionLanguage, -1, -1, UTF_8);

    assertEquals(PAYLOAD, getPayloadAsString(eventModel.getMessage(), UTF_8));
    assertEquals(ERROR_PAYLOAD, getPayloadAsString(eventModel.getError().getMessage(), UTF_8));
    assertEquals(ERROR_DESCRIPTION, eventModel.getError().getDescription());
    assertEquals(ERROR_DETAILED_DESCRIPTION, eventModel.getError().getDetailedDescription());
    assertEquals(CORE_NAMESPACE_NAME + ":" + ANY_IDENTIFIER, eventModel.getError().getType());
    assertEquals(RuntimeException.class.getName(), eventModel.getError().getExceptionType());
  }

  @Test
  public void getEventModelReturnsValueForJSONMimeType() {
    Message muleMessage = getMockedMuleMessage(DataType.JSON_STRING, PAYLOAD);

    EventModel eventModel = getEventModel(getEvent(muleMessage), mock(ExpressionLanguage.class), -1, -1, UTF_8);

    assertEquals(PAYLOAD, getPayloadAsString(eventModel.getMessage(), UTF_8));
  }

  @Test
  public void getEventModelReturnsValueForJSONMimeTypeUsingCharset() throws UnsupportedEncodingException {
    final MediaType mediaType = MediaType.create("application", "json", UTF_16);
    Message muleMessage = getMockedMuleMessage(DataType.builder().type(String.class).mediaType(mediaType).build(),
                                               PAYLOAD);

    EventModel eventModel = getEventModel(getEvent(muleMessage), mock(ExpressionLanguage.class), -1, -1, UTF_8);

    assertEquals(mediaType.toRfcString(), eventModel.getMessage().getPayload().getDataType().getMediaType());
    assertEquals(PAYLOAD, getPayloadAsString(eventModel.getMessage(), UTF_16));
  }


  @Test
  public void getEventModelReturnsNullContentTruncatedForInputStreamMimeType() {
    Message muleMessage = getMockedMuleMessage(DataType.INPUT_STREAM, new ByteArrayInputStream(PAYLOAD.getBytes()));

    EventModel eventModel = getEventModel(getEvent(muleMessage), mock(ExpressionLanguage.class), -1, -1, UTF_8);

    assertNull(eventModel.getMessage().getPayload().getContent());
    assertTrue(eventModel.getMessage().getPayload().isTruncated());
  }

  @Test
  public void getEventModelReturnsValueForCursorStreamProvider() {
    ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(PAYLOAD.getBytes());
    Message muleMessage =
        getMockedMuleMessage(DataType.CURSOR_STREAM_PROVIDER, new InMemoryCursorStreamProvider(byteArrayInputStream,
                                                                                               InMemoryCursorStreamConfig
                                                                                                   .getDefault(),
                                                                                               new PoolingByteBufferManager()));
    EventModel eventModel = getEventModel(getEvent(muleMessage), mock(ExpressionLanguage.class), -1, -1, UTF_8);

    assertEquals(PAYLOAD, getPayloadAsString(eventModel.getMessage(), UTF_8));
    assertFalse(eventModel.getMessage().getPayload().isTruncated());
  }

  @Test
  public void getEventModelHandlesCursorIteratorProvider() {
    CursorIteratorProvider cursorIteratorProvider = mock(CursorIteratorProvider.class);
    CursorIterator iterator = mock(CursorIterator.class);
    when(cursorIteratorProvider.openCursor()).thenReturn(iterator);
    AtomicInteger count = new AtomicInteger(CURSOR_ITERATOR_PAYLOADS);

    when(iterator.hasNext()).thenAnswer((invocationOnMock) -> count.getAndDecrement() > 0);
    when(iterator.next()).thenReturn(PAYLOAD);

    ArrayList expectedContent = new ArrayList();
    for (int i = 0; i < CURSOR_ITERATOR_PAYLOADS; i++) {
      expectedContent.add(PAYLOAD);
    }

    Message muleMessage = getMockedMuleMessage(DataType.CURSOR_ITERATOR_PROVIDER, cursorIteratorProvider);
    ExpressionLanguage expressionLanguage = getMockedExpressionLanguage(typedValue -> {
      CursorIterator openCursor = ((CursorIteratorProvider) typedValue.getValue()).openCursor();
      try {
        List<String> content = new ArrayList<>();
        while (openCursor.hasNext()) {
          content.add((String) openCursor.next());
        }
        return new TypedValue(content.toString(), STRING);
      } finally {
        try {
          openCursor.close();
        } catch (IOException e) {
        }
      }
    });

    EventModel eventModel = getEventModel(getEvent(muleMessage), expressionLanguage, -1, -1, UTF_8);
    assertEquals(expectedContent.toString(), getPayloadAsString(eventModel.getMessage(), UTF_8));
    assertFalse(eventModel.getMessage().getPayload().isTruncated());
  }

  @Test
  public void getEventModelReturnsStringForPojoMimeTypeWhenForced() {
    Message muleMessage = getMockedMuleMessage(STRING, STRING_PAYLOAD);
    ExpressionLanguage expressionLanguage = getMockedExpressionLanguage();

    EventModel eventModel = getEventModel(getEvent(muleMessage), expressionLanguage, -1, -1, UTF_8);

    assertEquals(STRING_PAYLOAD, getPayloadAsString(eventModel.getMessage(), UTF_8));
    assertEquals(STRING.getType().getName(), eventModel.getMessage().getPayload().getDataType().getType());
    assertEquals(DataType.builder(STRING).charset(UTF_8).build().getMediaType().toRfcString(),
                 eventModel.getMessage().getPayload().getDataType().getMediaType());
  }

  @Test
  public void getEventModelReturnsValueTruncatedForLargeCursorStreamProvider() {
    ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(BIG_PAYLOAD.getBytes());
    Message muleMessage =
        getMockedMuleMessage(DataType.CURSOR_STREAM_PROVIDER, new InMemoryCursorStreamProvider(byteArrayInputStream,
                                                                                               InMemoryCursorStreamConfig
                                                                                                   .getDefault(),
                                                                                               new PoolingByteBufferManager()));
    ExpressionLanguage expressionLanguage = mock(ExpressionLanguage.class);

    EventModel eventModel = getEventModel(getEvent(muleMessage), expressionLanguage, PAYLOAD.getBytes().length, -1, UTF_8);

    assertNull(eventModel.getMessage().getPayload().getContent());
    assertTrue(eventModel.getMessage().getPayload().isTruncated());
  }

  @Test
  public void getEventModelReturnsValueNotTruncatedForCursorStreamProvider() {
    ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(BIG_PAYLOAD.getBytes());
    Message muleMessage =
        getMockedMuleMessage(DataType.CURSOR_STREAM_PROVIDER, new InMemoryCursorStreamProvider(byteArrayInputStream,
                                                                                               InMemoryCursorStreamConfig
                                                                                                   .getDefault(),
                                                                                               new PoolingByteBufferManager()));
    ExpressionLanguage expressionLanguage = mock(ExpressionLanguage.class);

    EventModel eventModel =
        getEventModel(getEvent(muleMessage), expressionLanguage, BIG_PAYLOAD.getBytes().length + 1, -1, UTF_8);

    assertEquals(BIG_PAYLOAD, getPayloadAsString(eventModel.getMessage(), UTF_8));
    assertFalse(eventModel.getMessage().getPayload().isTruncated());
  }

  @Test
  public void getEventModelReturnsValueForNullMimeTypeIfForced() {
    Message muleMessage = getMockedMuleMessage(null, STRING_PAYLOAD);
    ExpressionLanguage expressionLanguage = mock(ExpressionLanguage.class);
    when(expressionLanguage.evaluate(anyString(), any(BindingContext.class)))
        .thenReturn((TypedValue) TypedValue.of(STRING_PAYLOAD));

    EventModel eventModel = getEventModel(getEvent(muleMessage), expressionLanguage, -1, -1, UTF_8);

    assertEquals(STRING_PAYLOAD, getPayloadAsString(eventModel.getMessage(), UTF_8));
  }

  @Test
  public void getEventModelReturnsEmptyPayloadForNullPayload() {
    Message muleMessage = getMockedMuleMessage(null, null);
    ExpressionLanguage expressionLanguage = mock(ExpressionLanguage.class);

    EventModel eventModel = getEventModel(getEvent(muleMessage), expressionLanguage, -1, -1, UTF_8);

    assertNull(eventModel.getMessage().getPayload().getContent());
  }

  @Test
  public void getEventModelReturnsSuccessfulEventModel() {
    Message muleMessage = getMockedMuleMessage(null, null);
    ExpressionLanguage expressionLanguage = mock(ExpressionLanguage.class);
    EventModel eventModel = getEventModel(getEvent(muleMessage), expressionLanguage, -1, -1, UTF_8);
    assertThat(eventModel.isSuccessful(), is(true));
  }

  @Test
  public void getEventModelReturnsUnsuccessfulEventModel() {
    Message muleMessage = getMockedMuleMessage(null, null);
    ExpressionLanguage expressionLanguage = mock(ExpressionLanguage.class);
    EventModel eventModel = getEventModel(getEventWithError(muleMessage), expressionLanguage, -1, -1, UTF_8);
    assertThat(eventModel.isSuccessful(), is(false));
  }

  private Event getEvent(Message message) {
    Event event = mock(Event.class, RETURNS_DEEP_STUBS);
    when(event.getMessage()).thenReturn(message);
    when(event.getError()).thenReturn(Optional.empty());
    return event;
  }

  private Event getEventWithError(Message message) {
    Event event = mock(Event.class, RETURNS_DEEP_STUBS);
    when(event.getMessage()).thenReturn(message);

    Error error = mock(Error.class);
    when(error.getErrorType()).thenReturn(mock(ErrorType.class));
    when(error.getCause()).thenReturn(mock(Throwable.class));
    when(event.getError()).thenReturn(Optional.of(error));
    return event;
  }

  private Message getMockedMuleMessage(DataType dataType, Object payload) {
    return getMockedMuleMessage(dataType, payload, null);
  }

  private Message getMockedMuleMessage(DataType dataType, Object payload, Object attributes) {
    Message mockedMuleMessage = mock(Message.class);
    TypedValue mockedPayload = new TypedValue(payload, dataType);
    when(mockedMuleMessage.getPayload()).thenReturn(mockedPayload);
    if (attributes != null) {
      TypedValue mockedAttributes = new TypedValue(attributes, DataType.fromObject(attributes));
      when(mockedMuleMessage.getAttributes()).thenReturn(mockedAttributes);
    }

    return mockedMuleMessage;
  }

  private Message getMockedMuleMessageWithTypedValue(DataType dataType, Object payload, TypedValue attributes) {
    Message mockedMuleMessage = mock(Message.class);
    TypedValue mockedPayload = new TypedValue(payload, dataType);
    when(mockedMuleMessage.getPayload()).thenReturn(mockedPayload);
    when(mockedMuleMessage.getAttributes()).thenReturn(attributes);
    return mockedMuleMessage;
  }

  private String getPayloadAsString(MessageModel message, Charset charset) {
    return new String(message.getPayload().getContent(), charset);
  }

  private ExpressionLanguage getMockedExpressionLanguage() {
    return getMockedExpressionLanguage(typedValue -> new TypedValue(typedValue.getValue().toString(), DataType.builder()
        .type(String.class).mediaType(MediaType.create("application", "dw")).build()));
  }

  private ExpressionLanguage getMockedExpressionLanguage(Function<TypedValue, TypedValue> transformation) {
    ExpressionLanguage expressionLanguage = mock(ExpressionLanguage.class);
    when(expressionLanguage.evaluate(anyString(), any(BindingContext.class))).thenAnswer(
                                                                                         (Answer<TypedValue>) invocationOnMock -> transformation
                                                                                             .apply(((BindingContext) invocationOnMock
                                                                                                 .getArguments()[1])
                                                                                                     .lookup(PAYLOAD_BINDING_CONTEXT_KEY)
                                                                                                     .orElseThrow(() -> new IllegalStateException("Missing typed value from binding context"))));
    return expressionLanguage;
  }

  private static class TestAttributes {

    private final String name;
    private final String fileName;
    private final long size;
    private final Map<String, List<String>> headers;

    public TestAttributes(String name, String fileName, long size, Map<String, List<String>> headers) {
      this.name = name;
      this.fileName = fileName;
      this.size = size;
      this.headers = headers;
    }

    @Override
    public String toString() {
      return ReflectionToStringBuilder.toString(this, SHORT_PREFIX_STYLE);
    }
  }

}
