package org.mule.tools.connectivity.jenkins.client;

import com.offbytwo.jenkins.JenkinsServer;
import com.offbytwo.jenkins.model.Job;
import org.mule.tools.connectivity.jenkins.client.retrievers.MuleVersionRetriever;
import org.mule.tools.connectivity.jenkins.client.retrievers.WebDavMuleVersionsRetriever;
import org.mule.tools.connectivity.jenkins.client.xmlBuilder.DevelopClassicPullRequestBuilderXmlConfigBuilder;
import org.mule.tools.connectivity.jenkins.client.xmlBuilder.DevelopClassicXmlConfigBuilder;
import org.mule.tools.connectivity.jenkins.client.xmlBuilder.DummyXmlConfigBuilder;
import org.mule.tools.connectivity.jenkins.client.xmlBuilder.MasterClassicXmlConfigBuilder;
import org.mule.tools.connectivity.jenkins.client.xmlBuilder.Mule4XmlConfigBuilder;
import org.mule.tools.connectivity.jenkins.client.xmlBuilder.MultiMuleXmlConfigBuilder;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;

public class PlanBuilder {

    // BRANCHES NAMES
    public static final String DEVELOP_BRANCH = "develop";
    public static final String MASTER_BRANCH = "master";
    public static final String MIGRATED_PREMIUM_SELECT = "Migrated - Premium.Select";
    public static final String MIGRATED_COMMUNITY_STANDARD = "Migrated - Community.Standard";
    private static final String SALESFORCE_CONNECTOR = "salesforce-connector";
    private static final String TEST_MAVEN_GOAL = "test";
    private static final String INTERNAL_ERROR_URL_IS_MALFORMED_ERROR_MSG = "Internal error. URL is malformed.";
    private static final String ALL_VIEW = "All";

    JenkinsServer jenkinsServer;
    MuleVersionRetriever muleVersionRetriever;
    String user;
    String password;
    String email;

    public PlanBuilder(String jenkinsUrl, String user, String email, String password) {
        this.user = user;
        this.email = email;
        this.password = password;
        this.muleVersionRetriever = WebDavMuleVersionsRetriever.getInstance(user, password);
        try {
            jenkinsServer = new JenkinsServer(new URI(jenkinsUrl), email, password);
        } catch (URISyntaxException e) {
            throw new RuntimeException(INTERNAL_ERROR_URL_IS_MALFORMED_ERROR_MSG, e);
        }
    }

    public PlanBuilder(JenkinsServer jenkinsServer, MuleVersionRetriever muleVersionRetriever){
        this.jenkinsServer = jenkinsServer;
        this.muleVersionRetriever = muleVersionRetriever;
    }

    public void createClassicPlans(ConnectorConfiguration connector) throws IOException {
        if (connector.getDevelopSupport()) {
            createDevelopClassicPlan(connector, TEST_MAVEN_GOAL);
        }
        if (connector.getMasterSupport()) {
            createMasterClassicPlan(connector, TEST_MAVEN_GOAL);
        }
    }

    public void createMultiMulePlans(ConnectorConfiguration connector) throws IOException {
        if (connector.getDevelopSupport()) {
            createMultiMulePlan(connector, DEVELOP_BRANCH);
        }
        if (connector.getMasterSupport()) {
            createMultiMulePlan(connector, MASTER_BRANCH);
        }
    }

    public void createMule4xPlan(ConnectorConfiguration connector) throws IOException {
        String name = "mule4-" + connector.getName() + "-" + DEVELOP_BRANCH;
        String viewName = "mule4";

        createOrUpdate(Mule4XmlConfigBuilder.buildXmlConfig(this.muleVersionRetriever, connector), name, viewName);
    }

    public void createDevelopClassicPlan(ConnectorConfiguration connector, String mavenGoal) throws IOException {
        String name = connector.getName() + "-" + DEVELOP_BRANCH;
        String viewName = connector.getIsEnterprise() || connector.getName().contains(SALESFORCE_CONNECTOR) ? MIGRATED_PREMIUM_SELECT : MIGRATED_COMMUNITY_STANDARD;
        String viewNamePRBuilder = "PR-Builder";
        String namePRBuilder = name + "-PR-BUILDER";

        createOrUpdate(DevelopClassicXmlConfigBuilder.buildXmlConfig(connector, mavenGoal), name, viewName);
        createOrUpdate(DevelopClassicPullRequestBuilderXmlConfigBuilder.buildXmlConfig(connector, mavenGoal), namePRBuilder, viewNamePRBuilder);
    }

    public void createMasterClassicPlan(ConnectorConfiguration connector, String mavenGoal) throws IOException {
        String name = connector.getName() + "-" + MASTER_BRANCH;
        String viewName = connector.getIsEnterprise() || connector.getName().contains(SALESFORCE_CONNECTOR) ? MIGRATED_PREMIUM_SELECT : MIGRATED_COMMUNITY_STANDARD;

        createOrUpdate(MasterClassicXmlConfigBuilder.buildXmlConfig(connector, mavenGoal), name, viewName);
    }

    public void createMultiMulePlan(ConnectorConfiguration connector, String branch) throws IOException {
        MultiMuleXmlConfigBuilder xmlConfigBuilder = new MultiMuleXmlConfigBuilder(muleVersionRetriever);
        String xml = xmlConfigBuilder.buildXmlConfig(connector, branch);
        String name = connector.getName() + "-multimule-" + branch;
        String viewName = connector.getIsEnterprise() || connector.getName().contains(SALESFORCE_CONNECTOR) ? MIGRATED_PREMIUM_SELECT : MIGRATED_COMMUNITY_STANDARD;

        createOrUpdate(xml, name, viewName);
    }

    public void createOrUpdate(String xml, String name, String viewName) throws IOException {
        if (jenkinsServer.getJobs().get(name.toLowerCase()) == null) {
            jenkinsServer.createJob(name, xml, true);
            System.out.println("Created: " + name);
            jenkinsServer.addJobToView(name, viewName, true);
        } else {
            System.out.println("Updated: " + name);
            jenkinsServer.updateJob(name, xml, true);
        }
    }

    public void createOrUpdate(String xml, String name) throws IOException {
        if (jenkinsServer.getJobs().get(name.toLowerCase()) == null) {
            jenkinsServer.createJob(name, xml, true);
            System.out.println("Created: " + name);
        } else {
            System.out.println("Updated: " + name);
            jenkinsServer.updateJob(name, xml, true);
        }
    }

    public void deletePlans(ConnectorConfiguration connector) throws IOException {
        List<Job> jobList = jenkinsServer.getView(ALL_VIEW).getJobs();
        for (Job job : jobList) {
            if (job.getName().startsWith(connector.getName())) {
                System.out.println("Deleting: " + job.getName());
                jenkinsServer.deleteJob(job.getName());
            }
        }
    }

    public void createBYOCDummyPlan(ConnectorConfiguration connector) {
        DummyXmlConfigBuilder dummyXmlConfigBuilder = new DummyXmlConfigBuilder();
        dummyXmlConfigBuilder.buildXmlConfig(connector);
    }

    public void deletePlan(String jobName) throws IOException {
        jenkinsServer.deleteJob(jobName);
    }
}
