package org.mule.tools.connectivity.jenkins.client.models.step;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.nio.file.Paths;

import static org.mule.tools.connectivity.jenkins.client.util.StringUtils.ESCAPED_NEWLINE;
import static org.mule.tools.connectivity.jenkins.client.util.StringUtils.reduceWithNewlines;

public class PropertiesFileStep implements JenkinsStep {

    public static final String DEFAULT_FILE_LOCATION = "src/test/resources/";
    public static final String DEFAULT_FILE_NAME = "automation-credentials.properties";
    private static final int MAX_CHAR_LENGTH = 80;

    public static class Builder {

        private final byte[] contents;
        private String location = DEFAULT_FILE_LOCATION;
        private String fileName = DEFAULT_FILE_NAME;

        public Builder(String contents) {
            this.contents = contents.getBytes();
        }

        public Builder(InputStream contents) throws IOException {
            this.contents = IOUtils.toByteArray(contents);
        }

        public Builder(File file) throws IOException {
            this.contents = IOUtils.toByteArray(new FileInputStream(file));
            this.fileName = file.getName();
        }

        public Builder withLocation(String location) {
            if(location != null)
                this.location = location;
            return this;
        }

        public Builder withFileName(String fileName) {
            if(fileName != null)
                this.fileName = fileName;
            return this;
        }

        public PropertiesFileStep build() {
            return new PropertiesFileStep(this);
        }

    }

    private byte[] contents;
    private String location = DEFAULT_FILE_LOCATION;
    private String fileName = DEFAULT_FILE_NAME;

    public PropertiesFileStep(String fileContents) {
        this.contents = fileContents.getBytes();
    }

    public PropertiesFileStep(File file) throws IOException {
        this.contents = IOUtils.toByteArray(new FileInputStream(file));
        this.fileName = file.getName();
    }

    public PropertiesFileStep(InputStream fileContents) throws IOException {
        this.contents = IOUtils.toByteArray(fileContents);
    }

    private PropertiesFileStep(Builder builder) {
        this.contents = builder.contents;
        this.fileName = builder.fileName;
        this.location = builder.location;
    }

    @Override
    public String getXmlConfig() {
        Path outputPath = Paths.get(location, fileName);
        String base64Contents = Base64.encodeBase64String(this.contents);

        String shellFormat = "echo \"%s\" | " + ESCAPED_NEWLINE + "base64 --decode > %s";
        String shellCommand = String.format(shellFormat, reduceWithNewlines(base64Contents, MAX_CHAR_LENGTH), outputPath.toString());
        return new ShellStep(shellCommand).getXmlConfig();
    }

}
