package org.mule.tools.connectivity.jenkins.client.util;

import static org.apache.commons.lang3.StringUtils.isBlank;

import org.mule.tools.connectivity.jenkins.client.ConnectorConfiguration;
import org.mule.tools.connectivity.jenkins.client.models.JenkinsModel;
import org.mule.tools.connectivity.jenkins.client.models.step.MavenStep;
import org.mule.tools.connectivity.jenkins.client.models.step.ShellStep;

import com.google.common.base.Objects;

/**
 * Created by estebanwasinger on 8/11/15.
 */
public class StepBuilder {

    public static final String BATCH_MODE = " -B";
    public static final String JACOCO_PREPARE_PHASE = " org.jacoco:jacoco-maven-plugin:0.7.2.201409121644:prepare-agent";
    public static final String SONAR_PHASE = " org.codehaus.mojo:sonar-maven-plugin:2.6:sonar";
    public static final String MAVEN_REPO_SETTINGS = " -Dmavenlocalrepo=/home/jenkins/.m2 -Dm2home=/opt/maven/latest";

    public static MavenStep classicDevelopBuildStep(String mavenGoal) {
        return new MavenStep(BATCH_MODE + JACOCO_PREPARE_PHASE + " " + mavenGoal + MAVEN_REPO_SETTINGS);
    }

    public static MavenStep classicDevelopBuildStep(final ConnectorConfiguration connector, final String mavenGoal) {
        return new MavenStep(BATCH_MODE + JACOCO_PREPARE_PHASE + " " + mavenGoal + " " + connector.getTestingConfiguration() + MAVEN_REPO_SETTINGS, connector.getJvmOptions());
    }

    public static MavenStep classicDevelopBuildStep(final ConnectorConfiguration connector, final String mavenGoal, final String activeConfig) {
        return new MavenStep(BATCH_MODE + JACOCO_PREPARE_PHASE + " " + mavenGoal + " -Dactiveconfiguration=" + activeConfig + " " + connector.getTestingConfiguration() + MAVEN_REPO_SETTINGS, connector.getJvmOptions());
    }

    public static MavenStep classicMasterBuildStep(final ConnectorConfiguration connector, final String mavenGoal) {
        return new MavenStep(BATCH_MODE + " clean " + mavenGoal + " " + Objects.firstNonNull(connector.getTestingConfigurationForMaster(), connector.getTestingConfiguration()) + " -Dmule.testingMode=true" + MAVEN_REPO_SETTINGS, connector.getJvmOptions());
    }

    public static MavenStep sonarStep(){
        return new MavenStep(BATCH_MODE + SONAR_PHASE + " -Dappend=true");
    }

    public static MavenStep newSonarStep(){
        return new MavenStep(BATCH_MODE + SONAR_PHASE + " -Dappend=true -Dsonar.host.url=https://sonar-tools.mulesoft.com/ -Dsonar.login=$SONAR_USERNAME -Dsonar.password=$SONAR_PASSWORD");
    }

    public static MavenStep newSonarPRBuilderStep(final ConnectorConfiguration connector) {
        final GithubSSHURLParser githubSSHURLParser = new GithubSSHURLParser(connector.getGitHubUrl());
        final String repoUrl = githubSSHURLParser.getGithubUser() + "/" + githubSSHURLParser.getGithubRepository();
        return new MavenStep(BATCH_MODE + SONAR_PHASE + " -Dsonar.host.url=https://sonar-tools.mulesoft.com/ -Dsonar.login=$SONAR_USERNAME -Dsonar.password=$SONAR_PASSWORD -Dsonar.analysis.mode=preview -Dsonar.github.oauth=$SONAR_GITHUB_OAUTH -Dsonar.github.pullRequest=$ghprbPullId " + "-Dsonar.github.repository=" + repoUrl);
    }

    public static MavenStep deployStep(){
        return new MavenStep(BATCH_MODE + " deploy -DskipTests");
    }

    public static MavenStep cleanStep() {
        return new MavenStep(BATCH_MODE + " clean");
    }

    public static MavenStep packageWithoutTestsStep() {
        return new MavenStep(BATCH_MODE + " clean package -Dmaven.test.skip");
    }

    public static MavenStep ctfBuildStep(final ConnectorConfiguration connector, final String muleVersion, String branch) {
        return ctfBuildStep(connector, muleVersion, branch, null);
    }

    public static MavenStep ctfBuildStep(final ConnectorConfiguration connector, final String muleVersion, String branch, String activeConfig) {
        final String testingConfig = Objects.firstNonNull("master".equals(branch) ? connector.getTestingConfigurationForMaster() : null, connector.getTestingConfiguration());
        final String activeConfigParameter = isBlank(activeConfig)? "" : " -Dactiveconfiguration=" + activeConfig;
        return new MavenStep("test " + testingConfig + activeConfigParameter + MAVEN_REPO_SETTINGS + " -Ddeploymentprofile=local -Dmuledirectory=" + connector.getOverrideMuleLocation() + muleVersion + " -Dsurefire.reportNameSuffix=" + muleVersion, connector.getJvmOptions());
    }

    public static MavenStep cloudHubBuildStep(final ConnectorConfiguration connector) {
        return new MavenStep("test " + connector.getTestingConfiguration() + MAVEN_REPO_SETTINGS + " -Ddeploymentprofile=cloudhub -Dcloudhubuser=${cloudhubuser} -Dcloudhubpassword=${cloudhubpassword} -Dsurefire.reportNameSuffix=cloudhub", connector.getJvmOptions());
    }

    public static ShellStep getEnableJceSteps() {
        return new ShellStep(
                "if [ \\! -d $HOME/jdk6jce ]\n" +
                "then\n" +
                "  cp -Lr /opt/jdk/jdk1.6.latest/ $HOME/jdk6jce\n" +
                "fi\n" +
                "cp -f /private/**/jce/jce6/*.jar $HOME/jdk6jce/jre/lib/security\n\n"
                        +
                "if [ \\! -d $HOME/jdk7jce ]\n" +
                "then\n" +
                "  cp -Lr /opt/jdk/jdk1.7.latest/ $HOME/jdk7jce\n" +
                "fi\n" +
                "cp -f /private/**/jce/jce7/*.jar $HOME/jdk7jce/jre/lib/security\n\n"
                        +
                "if [ \\! -d $HOME/jdk8jce ]\n" +
                "then\n" +
                "  cp -Lr /opt/jdk/jdk8.latest/ $HOME/jdk8jce\n" +
                "fi\n" +
                "cp -f /private/**/jce/jce8/*.jar $HOME/jdk8jce/jre/lib/security\n\n"
        );
    }

    public static ShellStep unzipMuleStandaloneStep(final String muleVersion) {
        return new ShellStep("if [ \\! -d " + muleVersion + " ]\n" +
                "then\n" +
                "unzip /private/mule-connectors/mules/" + muleVersion + ".zip\n" +
                "fi\n");
    }

    public static ShellStep getUpdateMuleConfForJceEnabledJavaSteps() {
        return new ShellStep(
                "sed -i.bak s?wrapper.java.command=/opt/jdk/jdk1.6.latest/bin/java?wrapper.java.command=$HOME/jdk6jce/jre/bin/java?g **/conf/wrapper.conf\n" +
                "sed -i.bak s?wrapper.java.command=/opt/jdk/jdk1.7.latest/bin/java?wrapper.java.command=$HOME/jdk7jce/jre/bin/java?g **/conf/wrapper.conf\n" +
                "sed -i.bak s?wrapper.java.command=/opt/jdk/jdk8.latest/bin/java?wrapper.java.command=$HOME/jdk8jce/jre/bin/java?g **/conf/wrapper.conf\n"
        );
    }

    public static String stepsToString(final JenkinsModel... steps){
        String stringSteps = "";
        for(JenkinsModel step : steps){
            stringSteps = stringSteps.concat(step.getXmlConfig());
        }
        return stringSteps;
    }
}
