package org.mule.tools.connectivity.jenkins.client.retrievers;

import com.github.sardine.DavResource;
import com.github.sardine.Sardine;
import com.github.sardine.SardineFactory;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class WebDavMuleVersionsRetriever extends MuleVersionRetriever {

    private static final String DEFAULT_WEBDAV_URL = "https://repository-mule-connectors.forge.cloudbees.com/private/mules/";

    private String user;
    private String password;
    private String webdavUrl;

    private static Map<String, List<String>> muleList = new HashMap<>();
    private static WebDavMuleVersionsRetriever instance = null;

    private Logger logger = LogManager.getLogger(WebDavMuleVersionsRetriever.class);


    /**
     * This is the singleton getInstance point. Don't use this, as it will be removed on future versions.
     * Manually create new instances instead.
     */
    @Deprecated
    public static WebDavMuleVersionsRetriever getInstance(String user, String password) {
        if (instance == null) {
            instance = new WebDavMuleVersionsRetriever(user, password);
        }

        return instance;
    }

    public WebDavMuleVersionsRetriever(String user, String password) {
        this.user = user;
        this.password = password;
        this.webdavUrl = DEFAULT_WEBDAV_URL;
    }

    public WebDavMuleVersionsRetriever(String webdavUrl, String user, String password) {
        this.user = user;
        this.password = password;
        this.webdavUrl = webdavUrl;
    }

    @Override
    public List<String> getEEMule3xList(String muleMinVersion) {
        return getEEMule3xList().stream()
                .filter(mule -> isValidMuleVersion(muleMinVersion, mule.split("-")[3]))
                .collect(Collectors.toList());
    }

    @Override
    public List<String> getCEMule3xList() {
        return getMuleList().stream()
                .filter(mule -> !is4xVersion(mule))
                .filter(mule -> !mule.toLowerCase().contains("enterprise"))
                .sorted()
                .collect(Collectors.toList());
    }

    @Override
    public List<String> getEEMule3xList() {
        return getMuleList().stream()
                .filter(mule -> !is4xVersion(mule))
                .filter(mule -> mule.toLowerCase().contains("enterprise"))
                .sorted()
                .collect(Collectors.toList());
    }

    @Override
    public List<String> getCEMule3xList(String muleMinVersion) {
        return getCEMule3xList().stream()
                .filter(mule -> isValidMuleVersion(muleMinVersion, mule.split("-")[2]))
                .collect(Collectors.toList());
    }

    @Override
    public List<String> getMule4xList() {
        return getMuleList().stream()
                .filter(mule -> is4xVersion(mule))
                .sorted()
                .collect(Collectors.toList());
    }

    private List<String> getMuleList() {
        if (muleList.containsKey(this.webdavUrl)) {
            logger.trace("Using already retrieved Mule runtime list.");
            return muleList.get(this.webdavUrl);
        }

        List<DavResource> resources;
        try {
            Sardine sardine = SardineFactory.begin(user, password);
            resources = sardine.getResources(webdavUrl);
            sardine.shutdown();
        } catch (IOException e) {
            throw new RuntimeException("Problem retrieving mule versions. Please check your credentials and ensure that are correct.");
        }

        List<String> muleVersionList = resources.stream()
                .filter(res -> res.getName().endsWith(".zip") && !res.getName().startsWith("._"))
                .map(res -> res.getName().substring(0, res.getName().length() - 4))
                .peek(res -> logger.trace("Found Mule: " + res))
                .collect(Collectors.toList());

        muleList.put(this.webdavUrl, muleVersionList);
        return muleVersionList;
    }

    public String getWebdavUrl() {
        return webdavUrl;
    }

    public void setWebdavUrl(String webdavUrl) {
        this.webdavUrl = webdavUrl;
    }

}
