package org.mule.tools.connectivity.jenkins.client.xmlBuilder;

import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.collections.CollectionUtils.isNotEmpty;
import static org.mule.tools.connectivity.jenkins.client.util.StepBuilder.cloudHubBuildStep;
import static org.mule.tools.connectivity.jenkins.client.util.StepBuilder.ctfCleanBuildStep;
import static org.mule.tools.connectivity.jenkins.client.util.StepBuilder.getEnableJceSteps;
import static org.mule.tools.connectivity.jenkins.client.util.StepBuilder.getUpdateMuleConfForJceEnabledJavaSteps;
import static org.mule.tools.connectivity.jenkins.client.util.StepBuilder.unzipMuleStandaloneStep;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.mule.tools.connectivity.jenkins.client.ConnectorConfiguration;
import org.mule.tools.connectivity.jenkins.client.models.properties.JobBlocker;
import org.mule.tools.connectivity.jenkins.client.models.publishers.JunitResultArchiver;
import org.mule.tools.connectivity.jenkins.client.models.step.MavenStep;
import org.mule.tools.connectivity.jenkins.client.models.step.ShellStep;
import org.mule.tools.connectivity.jenkins.client.models.triggers.TimeTrigger;
import org.mule.tools.connectivity.jenkins.client.retrievers.MuleVersionRetriever;
import org.mule.tools.connectivity.jenkins.client.util.GithubSSHURLParser;

import com.google.common.collect.Lists;

import static org.mule.tools.connectivity.jenkins.client.JobConstants.*;


public class MultiMuleXmlConfigBuilder extends BasicXmlConfigBuilder {

    private static final String CLEAN_PACKAGE_DSKIP_TESTS_GOAL = "clean package -DskipTests";

    MuleVersionRetriever muleVersionRetriever;

    public MultiMuleXmlConfigBuilder(final MuleVersionRetriever muleVersionRetriever) {
        this.muleVersionRetriever = muleVersionRetriever;
    }

    public String buildXmlConfig(final ConnectorConfiguration connector, final String branch) {

        String xml = getResourceAsString(BASIC_TEMPLATE_XML);

        String hackStep = StringUtils.EMPTY;
        if (!connector.getHackStep().equals(StringUtils.EMPTY)) {
            hackStep = new ShellStep(connector.getHackStep()).getXmlConfig();
        }

        Map<String, String> valuesMap = new HashMap<>();

        String cloudHubConfig = "";

        if (connector.getCloudHubEnabled() && branch.contains(DEVELOP_BRANCH)) {
            cloudHubConfig = cloudHubBuildStep(connector).getXmlConfig();
        }

        valuesMap.put(GITHUB_CONFIG, connector.getGitHubConfig(branch).getXmlConfig());
        valuesMap.put(GITHUB_HTTP_URL, GithubSSHURLParser.toHttpsURL(connector.getGitHubUrl()));
        valuesMap.put(UNZIP_MULE_SHELL_STEP, unzipMulesSteps(connector) + hackStep);
        valuesMap.put(CONNECTOR_STEPS, connector.getSteps());
        valuesMap.put(CTF_BUILD_STEP, ctfBuildSteps(connector, branch) + cloudHubConfig);
        valuesMap.put(MAILER, connector.getMailer().getXmlConfig());
        valuesMap.put(TRIGGERS, new TimeTrigger(branch.contains(DEVELOP_BRANCH) ? EVERY_SUNDAY : EVERY_15_DAYS).getXmlConfig());
        valuesMap.put(JDK_VERSION, connector.getJceEnabled() ? JDK_1_7_JCE : JDK_1_7);
        valuesMap.put(ENABLE_JCE_STEPS, connector.getJceEnabled() ? getEnableJceSteps().getXmlConfig() : StringUtils.EMPTY);
        valuesMap.put(MULE_CONF_FOR_JCE_STEPS, connector.getJceEnabled() ? getUpdateMuleConfForJceEnabledJavaSteps().getXmlConfig() : StringUtils.EMPTY);
        valuesMap.put(PROPERTIES, connector.getParallelBuilds() ? StringUtils.EMPTY : new JobBlocker(connector.getName() + ".*").getXmlConfig());
        valuesMap.put(BUILD_WRAPPERS, "");
        valuesMap.put(PUBLISHERS, new JunitResultArchiver().getXmlConfig());
        StrSubstitutor sub = new StrSubstitutor(valuesMap);
        return sub.replace(xml);
    }

    private String unzipMulesSteps(final ConnectorConfiguration connector) {
        String steps = StringUtils.EMPTY;
        for (String muleVersion : muleVersionRetriever.getEEMule3xList(connector.getMinMuleVersion())) {
            steps = steps.concat(unzipMuleStandaloneStep(muleVersion).getXmlConfig());
        }
        if (!connector.getIsEnterprise()) {
            for (String muleVersion : muleVersionRetriever.getCEMule3xList(connector.getMinMuleVersion())) {
                steps = steps.concat(unzipMuleStandaloneStep(muleVersion).getXmlConfig());
            }
        }
        return steps;
    }

    private String ctfBuildSteps(final ConnectorConfiguration connector, String branch) {
        List<MavenStep> steps = Lists.newArrayList(new MavenStep(CLEAN_PACKAGE_DSKIP_TESTS_GOAL));
        List<String> muleList = Stream.concat(muleVersionRetriever.getEEMule3xList(connector.getMinMuleVersion()).stream(),
                connector.getIsEnterprise() ? Stream.empty() : muleVersionRetriever.getCEMule3xList(connector.getMinMuleVersion()).stream()).collect(toList());

        if (isNotEmpty(connector.getActiveConfigs())) {
            connector.getActiveConfigs().forEach(activeConfig -> {
                muleList.forEach(muleVersion -> steps.add(ctfCleanBuildStep(connector, muleVersion, branch, activeConfig)));
            });
        } else {
            muleList.forEach(muleVersion -> steps.add(ctfCleanBuildStep(connector, muleVersion, branch)));
        }

        return steps.stream().map(step -> step.getXmlConfig()).collect(joining());
    }

}
