package org.mule.tools.connectivity.jenkins.client;

import com.offbytwo.jenkins.JenkinsServer;
import com.offbytwo.jenkins.model.FolderJob;
import com.offbytwo.jenkins.model.Job;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.mule.tools.connectivity.jenkins.client.xmlBuilder.DevelopClassicPullRequestBuilderXmlConfigBuilder;
import org.mule.tools.connectivity.jenkins.client.xmlBuilder.DevelopClassicXmlConfigBuilder;
import org.mule.tools.connectivity.jenkins.client.xmlBuilder.DummyXmlConfigBuilder;
import org.mule.tools.connectivity.jenkins.client.xmlBuilder.MasterClassicXmlConfigBuilder;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;

import static org.mule.tools.connectivity.jenkins.client.JobConstants.DEVELOP_BRANCH;
import static org.mule.tools.connectivity.jenkins.client.JobConstants.MASTER_BRANCH;


public class PlanBuilder {

    private static final String ENTERPRISE_CONNECTORS_VIEW = "Connectors - Premium-Select";
    private static final String COMMUNITY_CONNECTORS_VIEW = "Connectors - Community";
    private static final String INTERNAL_ERROR_URL_IS_MALFORMED_ERROR_MSG = "Internal error. URL is malformed.";
    private static final String ALL_VIEW = "All";

    private Logger logger = LogManager.getLogger(PlanBuilder.class);

    private JenkinsServer jenkinsServer;
    private String user;
    private String password;
    private String email;

    
    public PlanBuilder(String jenkinsUrl, String user, String email, String password) {
        this.user = user;
        this.email = email;
        this.password = password;
        try {
            jenkinsServer = new JenkinsServer(new URI(jenkinsUrl), email, password);
        } catch (URISyntaxException e) {
            throw new RuntimeException(INTERNAL_ERROR_URL_IS_MALFORMED_ERROR_MSG, e);
        }
    }

    public PlanBuilder(JenkinsServer jenkinsServer){
        this.jenkinsServer = jenkinsServer;
    }

    public void createClassicPlans(ConnectorConfiguration connector) throws IOException {
        if (connector.getDevelopSupport()) {
            createDevelopClassicPlan(connector);
        }
        if (connector.getMasterSupport()) {
            createMasterClassicPlan(connector);
        }
    }

    public void createDevelopClassicPlan(ConnectorConfiguration connector) throws IOException {
        String name = connector.getName() + "-" + DEVELOP_BRANCH;
        String viewName = connector.getIsEnterprise() ? ENTERPRISE_CONNECTORS_VIEW : COMMUNITY_CONNECTORS_VIEW;
        String viewNamePRBuilder = "PR-Builder";
        String namePRBuilder = name + "-PR-BUILDER";

        createOrUpdate(DevelopClassicXmlConfigBuilder.buildXmlConfig(connector), name, viewName);
        createOrUpdate(DevelopClassicPullRequestBuilderXmlConfigBuilder.buildXmlConfig(connector), namePRBuilder, viewNamePRBuilder);
    }

    public void createMasterClassicPlan(ConnectorConfiguration connector) throws IOException {
        String name = connector.getName() + "-" + MASTER_BRANCH;
        String viewName = connector.getIsEnterprise() ? ENTERPRISE_CONNECTORS_VIEW : COMMUNITY_CONNECTORS_VIEW;

        createOrUpdate(MasterClassicXmlConfigBuilder.buildXmlConfig(connector), name, viewName);
    }

    public void createOrUpdate(String xml, String name, String viewName) throws IOException {
        if (jenkinsServer.getJobs().get(name.toLowerCase()) == null) {
            jenkinsServer.createJob(name, xml, true);
            logger.info("Created: " + name);
            jenkinsServer.addJobToView(name, viewName, true);
        } else {
            logger.info("Updated: " + name);
            jenkinsServer.updateJob(name, xml, true);
        }
    }

    public void createOrUpdate(String xml, String name, FolderJob folderJob) throws IOException {
        if (jenkinsServer.getJobs().get(name.toLowerCase()) == null) {
            jenkinsServer.createJob(folderJob, name, xml, true);
            logger.info("Created: " + name);
        } else {
            logger.info("Updated: " + name);
            jenkinsServer.updateJob(folderJob, name, xml, true);
        }
    }

    public void createOrUpdate(String xml, String name) throws IOException {
        if (jenkinsServer.getJobs().get(name.toLowerCase()) == null) {
            jenkinsServer.createJob(name, xml, true);
            logger.info("Created: " + name);
        } else {
            logger.info("Updated: " + name);
            jenkinsServer.updateJob(name, xml, true);
        }
    }

    public void deletePlans(ConnectorConfiguration connector) throws IOException {
        List<Job> jobList = jenkinsServer.getView(ALL_VIEW).getJobs();
        for (Job job : jobList) {
            if (job.getName().startsWith(connector.getName())) {
                logger.info("Deleting: " + job.getName());
                jenkinsServer.deleteJob(job.getName());
            }
        }
    }

    public void createBYOCDummyPlan(ConnectorConfiguration connector) {
        DummyXmlConfigBuilder dummyXmlConfigBuilder = new DummyXmlConfigBuilder();
        dummyXmlConfigBuilder.buildXmlConfig(connector);
    }

    public void deletePlan(String jobName) throws IOException {
        jenkinsServer.deleteJob(jobName);
    }

    public void deletePlan(FolderJob folder, String jobName) throws IOException {
        jenkinsServer.deleteJob(folder, jobName);
    }
}
