package org.mule.tools.connectivity.sonar.client.rest.client;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.io.ByteStreams;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.HttpResponse;
import org.apache.http.auth.AuthScheme;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.AuthState;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.protocol.ClientContext;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.ExecutionContext;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Map;

import static com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES;

public class SonarHttpClient {

    private CloseableHttpClient client;
    private URI uri;
    private HttpContext localContext;
    private ObjectMapper mapper;

    public SonarHttpClient(String baseUrl, String username, String password) {
        if (baseUrl.endsWith("/")) {
            baseUrl += "/";
        }

        try {
            this.uri = new URI(baseUrl + "api/");
        } catch (URISyntaxException e) {
            e.printStackTrace();
            throw new RuntimeException("Invalid base URL: " + baseUrl);
        }

        HttpClientBuilder builder = addAuthentication(HttpClientBuilder.create(), this.uri, username, password);
        this.client = builder.build();

        this.localContext = new BasicHttpContext();
        this.localContext.setAttribute("preemptive-auth", new BasicScheme());

        this.mapper = this.getDefaultMapper();
    }

    private static HttpClientBuilder addAuthentication(HttpClientBuilder builder, URI uri, String username, String password) {
        CredentialsProvider provider = new BasicCredentialsProvider();
        AuthScope scope = new AuthScope(uri.getHost(), uri.getPort(), "realm");
        UsernamePasswordCredentials credentials = new UsernamePasswordCredentials(username, password);
        provider.setCredentials(scope, credentials);
        builder.setDefaultCredentialsProvider(provider);

        builder.addInterceptorFirst(new PreemptiveAuth());

        return builder;
    }


    static class PreemptiveAuth implements HttpRequestInterceptor {

        @Override
        public void process(HttpRequest request, HttpContext context) throws HttpException, IOException {
            AuthState authState = (AuthState) context.getAttribute(ClientContext.TARGET_AUTH_STATE);

            if (authState.getAuthScheme() == null) {
                AuthScheme authScheme = (AuthScheme) context.getAttribute("preemptive-auth");
                CredentialsProvider credsProvider = (CredentialsProvider) context.getAttribute(ClientContext.CREDS_PROVIDER);
                HttpHost targetHost = (HttpHost) context.getAttribute(ExecutionContext.HTTP_TARGET_HOST);
                if (authScheme != null) {
                    Credentials creds = credsProvider.getCredentials(
                            new AuthScope(targetHost.getHostName(), targetHost.getPort()));
                    if (creds == null) {
                        throw new HttpException("No credentials for preemptive authentication");
                    }
                    authState.update(authScheme, creds);
                }
            }
        }
    }

    private ObjectMapper getDefaultMapper() {
        ObjectMapper mapper = new ObjectMapper();
        mapper.disable(FAIL_ON_UNKNOWN_PROPERTIES);
        return mapper;
    }

    public <T> T get(String path, Map<String, String> queryParams, Class<T> clazz) throws IOException, URISyntaxException {
        URIBuilder builder = new URIBuilder(this.getPathURI(path));
        queryParams.forEach((k, v) -> builder.setParameter(k, v));
        return this.get(builder.build(), clazz);
    }

    public <T> T get(String path, Class<T> clazz) throws IOException {
        return this.get(this.getPathURI(path), clazz);
    }

    public <T> T get(URI uri, Class<T> clazz) throws IOException {
        HttpGet getMethod = new HttpGet(uri);
        HttpResponse response = client.execute(getMethod, localContext);
        try {
            return objectFromResponse(clazz, response);
        } finally {
            EntityUtils.consume(response.getEntity());
            releaseConnection(getMethod);
        }
    }

    private <T> T objectFromResponse(Class<T> clazz, HttpResponse response) throws IOException {
        InputStream content = response.getEntity().getContent();
        byte[] bytes = ByteStreams.toByteArray(content);
        List<T> result = mapper.readValue(bytes, mapper.getTypeFactory().constructCollectionType(List.class, clazz));
        return result.get(0);
    }

    private void releaseConnection(HttpRequestBase httpRequestBase) {
        httpRequestBase.releaseConnection();
    }

    private URI getPathURI(String path) {
        return this.uri.resolve(path);
    }

}
