package org.mule.tools.connectivity.sonar.client.rest;

import org.mule.tools.connectivity.sonar.client.rest.client.SonarHttpClient;
import org.mule.tools.connectivity.sonar.client.rest.model.Project;
import org.mule.tools.connectivity.sonar.client.rest.model.ProjectIssueSeverity;
import org.mule.tools.connectivity.sonar.client.rest.model.ProjectIssues;
import org.mule.tools.connectivity.sonar.client.rest.model.ProjectStatus;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static java.util.stream.Collectors.joining;


public class SonarRestClient {

    private static final String RESOURCE_PARAM = "resource";
    private static final String RESOURCES_ENDPOINT = "resources";
    private static final String DASHBOARD_INDEX_ENDPOINT = "dashboard/index/";
    private static final String ISSUES_SEARCH_ENDPOINT = "issues/search";

    private static final String ALERT_STATUS_METRIC_NAME = "alert_status";

    private static final String METRICS_PARAM = "metrics";
    private static final String INCLUDE_ALERTS_PARAM = "includealerts";
    private static final String COMPONENT_KEYS_PARAM = "componentKeys";
    private static final String SORT_PARAM = "s";
    private static final String ASCENDING_ORDER_PARAM = "asc";
    private static final String PAGE_SIZE_PARAM = "pageSize";
    private static final String SEVERITIES_PARAM = "severities";
    private static final String RESOLVED_PARAM = "resolved";

    private SonarHttpClient client;
    private String baseUrl;

    public SonarRestClient(String baseUrl, String username, String password) {
        this.client = new SonarHttpClient(baseUrl, username, password);
        this.baseUrl = baseUrl;
    }

    public String getProjectUrl(String projectKey) {
        return baseUrl + DASHBOARD_INDEX_ENDPOINT + projectKey;
    }

    public Project getProject(String projectKey) throws Exception {
        return client.getList(RESOURCES_ENDPOINT, params("resource", projectKey, "includetrends", "true"), Project.class).get(0);
    }

    public List<Map<String, String>> getProjectMetrics(String projectKey, String ... metrics) throws Exception {
        List<Project> projects = client.getList(RESOURCES_ENDPOINT,
                params(RESOURCE_PARAM, projectKey,
                       METRICS_PARAM, String.join(",", metrics)),
                Project.class);
        return projects.get(0).getMsr();
    }

    public ProjectStatus getProjectStatus(String projectKey) throws Exception {
        List<Project> projects = client.getList(RESOURCES_ENDPOINT,
                params(RESOURCE_PARAM, projectKey,
                       METRICS_PARAM, ALERT_STATUS_METRIC_NAME,
                       INCLUDE_ALERTS_PARAM, "true"),
                Project.class);
        return new ProjectStatus(takeByKey(projects.get(0).getMsr(), ALERT_STATUS_METRIC_NAME));
    }

    public ProjectIssues getProjectIssues(String projectKey) throws Exception {
        return getProjectIssues(projectKey, null, Collections.emptyList());
    }

    public ProjectIssues getProjectIssues(String projectKey, Integer maxResults) throws Exception {
        Map<String, String> params = new HashMap<>();
        params.put(COMPONENT_KEYS_PARAM, projectKey);
        params.put(SORT_PARAM, "SEVERITY");
        params.put(ASCENDING_ORDER_PARAM, "false");
        params.put(RESOLVED_PARAM, "false");

        if(maxResults != null)
            params.put(PAGE_SIZE_PARAM, maxResults.toString());

        return client.get(ISSUES_SEARCH_ENDPOINT, params, ProjectIssues.class);
    }

    public ProjectIssues getProjectIssues(String projectKey, Integer maxResults, ProjectIssueSeverity severity) throws Exception {
        return getProjectIssues(projectKey, maxResults, Collections.singletonList(severity));
    }

    public ProjectIssues getProjectIssues(String projectKey, Integer maxResults, List<ProjectIssueSeverity> severities) throws Exception {
        Map<String, String> params = new HashMap<>();
        params.put(COMPONENT_KEYS_PARAM, projectKey);
        params.put(SORT_PARAM, "SEVERITY");
        params.put(ASCENDING_ORDER_PARAM, "false");
        params.put(RESOLVED_PARAM, "false");

        if(severities != null && severities.size() > 0)
            params.put(SEVERITIES_PARAM, severities.stream().map(ProjectIssueSeverity::toString).collect(joining(",")));

        if(maxResults != null)
            params.put(PAGE_SIZE_PARAM, maxResults.toString());

        return client.get(ISSUES_SEARCH_ENDPOINT, params, ProjectIssues.class);
    }

    private Map<String, String> params(String... keyValuePairs) {
        Map<String, String> parameters = new HashMap<>();
        for(int i=0; i < keyValuePairs.length; i += 2) {
            parameters.put(keyValuePairs[i], keyValuePairs[i+1]);
        }
        return parameters;
    }

    private <K, V> Map<K, V> takeByKey(List<Map<K, V>> map, K k) {
        Optional<Map<K, V>> result = map.stream().filter(m -> m.get("key").equals(k)).findFirst();
        if(!result.isPresent()) {
            throw new RuntimeException("No entry found with the given key.");
        }
        return result.get();
    }
}
