package org.mule.tools.connectivity.sonar.client.rest.client;

import static com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.io.ByteStreams;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.HttpResponse;
import org.apache.http.auth.AuthScheme;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.AuthState;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.protocol.ClientContext;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.ExecutionContext;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;

public class SonarHttpClient {

    private CloseableHttpClient client;
    private URI uri;
    private HttpContext localContext;
    private ObjectMapper mapper;

    public SonarHttpClient(String baseUrl, String username, String password) {
        if (!baseUrl.endsWith("/")) {
            baseUrl += "/";
        }

        try {
            this.uri = new URI(baseUrl + "api/");
        } catch (URISyntaxException e) {
            throw new RuntimeException("Invalid base URL: " + baseUrl, e);
        }

        this.client = addAuthentication(HttpClientBuilder.create(), this.uri, username, password).build();

        this.localContext = new BasicHttpContext();
        this.localContext.setAttribute("preemptive-auth", new BasicScheme());

        this.mapper = this.getDefaultMapper();
    }

    private static HttpClientBuilder addAuthentication(HttpClientBuilder builder, URI uri, String username, String password) {
        CredentialsProvider provider = new BasicCredentialsProvider();
        AuthScope scope = new AuthScope(uri.getHost(), uri.getPort(), "realm");
        UsernamePasswordCredentials credentials = new UsernamePasswordCredentials(username, password);
        provider.setCredentials(scope, credentials);
        builder.setDefaultCredentialsProvider(provider);

        builder.addInterceptorFirst(new PreemptiveAuth());

        return builder;
    }


    static class PreemptiveAuth implements HttpRequestInterceptor {

        @Override
        public void process(HttpRequest request, HttpContext context) throws HttpException, IOException {
            AuthState authState = (AuthState) context.getAttribute(ClientContext.TARGET_AUTH_STATE);

            if (authState.getAuthScheme() == null) {
                AuthScheme authScheme = (AuthScheme) context.getAttribute("preemptive-auth");
                CredentialsProvider credsProvider = (CredentialsProvider) context.getAttribute(ClientContext.CREDS_PROVIDER);
                HttpHost targetHost = (HttpHost) context.getAttribute(ExecutionContext.HTTP_TARGET_HOST);
                if (authScheme != null) {
                    Credentials creds = credsProvider.getCredentials(
                            new AuthScope(targetHost.getHostName(), targetHost.getPort()));
                    if (creds == null) {
                        throw new HttpException("No credentials for preemptive authentication");
                    }
                    authState.update(authScheme, creds);
                }
            }
        }
    }

    private ObjectMapper getDefaultMapper() {
        ObjectMapper mapper = new ObjectMapper();
        mapper.disable(FAIL_ON_UNKNOWN_PROPERTIES);
        return mapper;
    }

    public <T> List<T> getList(String path, Map<String, String> queryParams, Class<T> tClass) throws IOException, URISyntaxException {
        return get(path, queryParams, getListType(tClass));
    }

    public <T> T get(String path, Map<String, String> queryParams, Class<T> tClass) throws IOException, URISyntaxException {
        return get(path, queryParams, getType(tClass));
    }

    private <T> T get(String path, Map<String, String> queryParams, JavaType type) throws IOException, URISyntaxException {
        URIBuilder builder = new URIBuilder(this.getPathURI(path));
        queryParams.forEach((k, v) -> builder.setParameter(k, v));
        HttpGet getMethod = new HttpGet(builder.build());
        HttpResponse response = client.execute(getMethod, localContext);
        try {
            return objectFromResponse(response, type);
        } finally {
            EntityUtils.consume(response.getEntity());
            releaseConnection(getMethod);
        }
    }

    private <T> T objectFromResponse(HttpResponse response, JavaType type) throws IOException {
        InputStream content = response.getEntity().getContent();
        byte[] bytes = ByteStreams.toByteArray(content);
        return mapper.readValue(bytes, type);
    }

    private void releaseConnection(HttpRequestBase httpRequestBase) {
        httpRequestBase.releaseConnection();
    }

    private URI getPathURI(String path) {
        return this.uri.resolve(path);
    }

    private JavaType getListType(Class clazz) {
        return mapper.getTypeFactory().constructCollectionType(List.class, clazz);
    }

    private JavaType getType(Class clazz) {
        return mapper.getTypeFactory().constructType(clazz);
    }

}
