package org.mule.tools.automationtestcoverage.inspectors;

import org.mule.tools.automationtestcoverage.model.ConnectorProcessorInfo;

import javax.annotation.processing.Messager;
import javax.annotation.processing.RoundEnvironment;
import javax.lang.model.element.*;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.mule.tools.automationtestcoverage.utils.Utils.throwError;

/**
 * This class handles the inspection of the /@Connector class
 *
 */
public class ConnectorInspector {

	private Messager messager;
	private RoundEnvironment environment;
	
	public ConnectorInspector(Messager messager, RoundEnvironment environment) {
		super();
		this.messager = messager;
		this.environment = environment;
	}
	
	/** 
	 * Register all /@Process methods from /@Connector class
	 * Get the name of the connector to find it later on the xmls
	 * Get all the methods annotated as /@Processor to know all the operations of the connector
	 * Populate ConnectorInfo
	 */
	public List<ConnectorProcessorInfo> inspectProcessors(TypeElement connectorAnnotationType) {
		
		List<ConnectorProcessorInfo> listConnectorProcessorInfo = new LinkedList<ConnectorProcessorInfo>();
		
		Set<? extends Element> elementsAnnotatedWith = environment.getElementsAnnotatedWith(connectorAnnotationType);
		
		if (elementsAnnotatedWith == null || elementsAnnotatedWith.isEmpty()) {
			throwError(messager, "Cannot locate any class annotated as @Connector");
		}
		
		// Connector Element Class
		for (Element element : elementsAnnotatedWith) {
			// Inspect only the methods
			if (element.getKind().equals(ElementKind.METHOD)) {				
				listConnectorProcessorInfo.add(new ConnectorProcessorInfo(element.getSimpleName().toString()));

			}
		}			
		
		return listConnectorProcessorInfo;
	}
	
	/**
	 * It will inspect all the classes annotated as /@Connector and will return a {@link List<String>} of all the
	 * configElementNames. One element name per connector.
	 * 
	 * @param connectorAnnotationType The element type of the /@Connector annotation
	 * @return List of config element names
	 */
	public List<String> inspectConnectors(TypeElement connectorAnnotationType) {
		
		List<String> connectorPrefixes = new LinkedList<String>();
		Set<? extends Element> elementsAnnotatedWith = environment.getElementsAnnotatedWith(connectorAnnotationType);
		
		if (elementsAnnotatedWith == null || elementsAnnotatedWith.isEmpty()) {
			throwError(messager, "Cannot locate any class annotated as @Connector");
		}
		
		// Connector Element Class
		for (Element element : elementsAnnotatedWith) {
			
			// Get all the annotations in the element
			List<? extends AnnotationMirror> annotationMirrors = element.getAnnotationMirrors();
			
			for (AnnotationMirror annotationMirror : annotationMirrors) {
				
				// Filter the @Connector annotation
				if (annotationMirror.getAnnotationType().equals(connectorAnnotationType.asType())) {
					
					// Get all the properties in the annotation
					Map<? extends ExecutableElement, ? extends AnnotationValue> elementValues = annotationMirror.getElementValues();
					
					for (ExecutableElement executableElement : elementValues.keySet()) {
						if (executableElement.getSimpleName().toString().equalsIgnoreCase("name")) {
							AnnotationValue annotationValue = elementValues.get(executableElement);
							connectorPrefixes.add(annotationValue.getValue().toString());
						}
					}
				}
			}
		}
		
		return connectorPrefixes;
	}
}
