package org.mule.tools.automationtestcoverage.inspectors;

import org.apache.log4j.Logger;
import org.mule.tools.automationtestcoverage.model.FlowInfo;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import javax.annotation.processing.Messager;
import javax.annotation.processing.RoundEnvironment;
import javax.lang.model.element.TypeElement;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.*;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static org.mule.tools.automationtestcoverage.utils.Utils.throwError;

public class MuleFlowInspector {

	static final private String testResourcesRelativePath = "src/test/resources";
	static private FilenameFilter filter = null;
	
	synchronized static private FilenameFilter getFilterInstance() {
		if (filter == null) {
			filter = new FilenameFilter() {
				
				@Override
				public boolean accept(File dir, String name) {
					return name.contains(".xml");
				}
			};
		}
		return filter;
	}
	
	private Messager messager;
    private Logger logger = Logger.getLogger(MuleFlowInspector.class);

    @SuppressWarnings("unused")
	private RoundEnvironment environment;
	private String projectPath;
	
	public MuleFlowInspector(Messager messager, RoundEnvironment environment, String projectPath) {
		super();
		this.messager = messager;
		this.environment = environment;
		this.projectPath = projectPath;
	}

	/** 
	 * Scan test/resources for xmls that has mule flows in them (if only one connector, there should be only one flow)
	 * Scan each xml and look up the flows
	 * Inspect the flow and check which operation is calling each flow
	 * Populate MuleFlowInfo
	 */
	public List<FlowInfo> inspectTestResourcesForFlowOperations(TypeElement e, String connectorXmlPrefix) {

		List<FlowInfo> listFlowInfo = new LinkedList<FlowInfo>();
		
		File testResourcesFolder = new File(projectPath + testResourcesRelativePath);
		
		if (testResourcesFolder == null || !testResourcesFolder.isDirectory()) {
			throwError(messager, "Cannot locate test resources folder in " + projectPath + testResourcesRelativePath);
		}

		for (File f : testResourcesFolder.listFiles(MuleFlowInspector.getFilterInstance())) {
            logger.debug("Inspecting for flows in " + f.getAbsolutePath());
			List<FlowInfo> inspectFileForFlows = null;
			try {
				inspectFileForFlows = MuleFlowInspector.inspectFileForFlows(f, connectorXmlPrefix);
				
				if (inspectFileForFlows != null && !inspectFileForFlows.isEmpty()) {
					for (FlowInfo fi : inspectFileForFlows) {
						listFlowInfo.add(fi);
					}
				}
			} catch (RuntimeException ex) {
				throwError(messager, ex.getMessage());
			}
		}
		
		if (listFlowInfo.isEmpty()) {
			throwError(messager, "None of the inspected XML files had a Mule Flow defined");
		}
		
		return listFlowInfo;
	}
	
	static public List<FlowInfo> inspectFileForFlows(File muleFlowFile, String connectorXmlPrefix) {
		List<FlowInfo> flowsInfo = new LinkedList<FlowInfo>();
		
		DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
		
		DocumentBuilder dBuilder = null;
		try {
			dBuilder = dbFactory.newDocumentBuilder();
		} catch (ParserConfigurationException e) {
			throw new RuntimeException("Cannot generate document building for file:  " + muleFlowFile.getAbsolutePath(), e);
		}
		
		Document doc = null;
		try {
			doc = dBuilder.parse(muleFlowFile);
		} catch (SAXException e) {
			throw new RuntimeException("Cannot parse the document for file: " + muleFlowFile.getAbsolutePath(), e);
		} catch (IOException e) {
			throw new RuntimeException("Cannot find the document building for file: " + muleFlowFile.getAbsolutePath(), e);
		}
		
		XPath xpath = XPathFactory.newInstance().newXPath();
		XPathExpression xPathExpression = null;
		
		try {
			xPathExpression = xpath.compile("/mule/flow");
		} catch (XPathExpressionException e) {
			assert false : "The xpathExpression cannot be compiled";
		}
		
		NodeList nodeList = null;
		try {
			nodeList = (NodeList) xPathExpression.evaluate(doc, XPathConstants.NODESET);
		} catch (XPathExpressionException e) {
			throw new RuntimeException("Cannot evaluate the document for file: " + muleFlowFile.getAbsolutePath(), e);
		}
		
		if (nodeList == null || nodeList.getLength() == 0) {
			return null;
		}
		
		// Iterate all the <mule><flow> elements
		
		for (int x = 0, xMax = nodeList.getLength() ; x < xMax ; x++ ) {
			
			FlowInfo flowInfo = new FlowInfo();
			
			// Get the name of the flow <flow name="myName">
			
			Node item = nodeList.item(x);
			NamedNodeMap attributes = item.getAttributes();
			flowInfo.setFlowName(attributes.getNamedItem("name").getNodeValue());
			
			// Get all the nodes inside the flow node <mule><flow>...
			
			XPathExpression xPathExpression2 = null;
			NodeList muleNodes = null;
			try {
				xPathExpression2 = xpath.compile("*");
				muleNodes = (NodeList) xPathExpression2.evaluate(item, XPathConstants.NODESET);
			} catch (XPathExpressionException e) {
				throw new RuntimeException("Cannot evaluate nodes inside flow with name " + flowInfo.getFlowName(), e);
			}
			
			Pattern muleOpPattern = Pattern.compile(connectorXmlPrefix + ":(.*)");
			
			if (muleNodes != null && muleNodes.getLength() > 0) {
				for (int y = 0, yMax = muleNodes.getLength() ; y < yMax ; y++ ) {
					Node muleNode = muleNodes.item(y);
					Matcher muleOpMatcher = muleOpPattern.matcher(muleNode.getNodeName());
					
					if (muleOpMatcher.find()) {
						flowInfo.addConnectorOperation(muleOpMatcher.group(1));
					}
				}
			}
			
			flowsInfo.add(flowInfo);
		}
		
		return flowsInfo;
	}
}
