package org.mule.tools.automationtestcoverage.inspectors;

import org.apache.log4j.Logger;
import org.mule.tools.automationtestcoverage.model.TestSourceItemInfo;
import org.mule.tools.automationtestcoverage.utils.Utils;

import javax.annotation.processing.Messager;
import javax.annotation.processing.RoundEnvironment;
import javax.lang.model.element.TypeElement;
import java.io.*;
import java.nio.charset.Charset;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static org.mule.tools.automationtestcoverage.utils.Utils.throwError;

public class TestSourcesInspector {
	
	static final private String testSourcesRelativePath = "src/test/java";	
	static final private Pattern lookUpFlowPatter = Pattern.compile("lookupMessageProcessor\\(\"([^\"]+)\"\\)");
	static final private Pattern lookUpFlowPatter2 = Pattern.compile("runFlowAndGetPayload\\(\"([^\"]+)\"\\)");
	
	
	static private FilenameFilter filter = null;
	
	synchronized static private FilenameFilter getFilterInstance() {
		if (filter == null) {
			filter = new FilenameFilter() {
				
				@Override
				public boolean accept(File dir, String name) {
					return name.contains(".java");
				}
			};
		}
		return filter;
	}
	
	private Messager messager;
    private Logger logger = Logger.getLogger(TestSourcesInspector.class);

    @SuppressWarnings("unused")
	private RoundEnvironment environment;
	private String projectPath;
	
	public TestSourcesInspector(Messager messager, RoundEnvironment environment, String projectPath) {
		super();
		this.messager = messager;
		this.environment = environment;
		this.projectPath = projectPath;
	}
	
	
	/** 
	 * Scan test/src for all the files annotated as .java
	 * Look for all the references to the operation lookupMessageProcessor(flowName) and relieve the flowName
	 * Map the flowName with all the operations that it calls
	 * Register all the operations as covered
	 */
	public List<TestSourceItemInfo> inspectTestSourcesForCallsToFlows(TypeElement e, String connectorXmlPrefix) {

		List<TestSourceItemInfo> listTestSourceItemInfo = new LinkedList<TestSourceItemInfo>();
		
		File testResourcesFolder = new File(projectPath + testSourcesRelativePath);
		
		if (testResourcesFolder == null || !testResourcesFolder.isDirectory()) {
			throwError(messager, "Cannot locate test resources folder in " + projectPath + testSourcesRelativePath);
		}
		
		recursiveInspectTestSources(e, connectorXmlPrefix, listTestSourceItemInfo, testResourcesFolder);

		return listTestSourceItemInfo;
	}
	
	public void recursiveInspectTestSources(TypeElement e, String connectorXmlPrefix, List<TestSourceItemInfo> listTestSourceItemInfo, File currentDirectory) {
		
		for (File f : currentDirectory.listFiles(TestSourcesInspector.getFilterInstance())) {		
			logger.debug("Inspecting for java test sources in " + f.getAbsolutePath());

			TestSourceItemInfo testSourceItemInfo = new TestSourceItemInfo(Utils.getRelativePath(projectPath, f.getAbsolutePath()));
			try {
				addOperationsInFileToTestSourceItemInfo(f, testSourceItemInfo);
				listTestSourceItemInfo.add(testSourceItemInfo);
			} catch (IOException e1) {
				Utils.throwError(messager, "Error during inspection for flowCalls in java file: " + f.getAbsolutePath());
			}
		}
		
		// Recursively iterate througth the inner directories
		for (File f : currentDirectory.listFiles()) {
			if (f.isDirectory()) {
				recursiveInspectTestSources(e, connectorXmlPrefix, listTestSourceItemInfo, f);
			}
		}
	}
	
	public TestSourceItemInfo addOperationsInFileToTestSourceItemInfo(File inputFile, final TestSourceItemInfo testSourceItemInfo) throws IOException {
				
		InputStream fis = new FileInputStream(inputFile);
		BufferedReader br = new BufferedReader(new InputStreamReader(fis, Charset.forName("UTF-8")));
		
		String line = null;
		Matcher matcher = null;
        boolean parseNext = false;
        int bracketCount = 0;

		while ((line = br.readLine()) != null) {

            if(line.contains("@Test")){
                parseNext = true;
                bracketCount = 0;

            }else if(parseNext && line.contains("{") && !line.contains("}")){
                bracketCount++;

            }else if(parseNext && line.contains("}") && !line.contains("{")){
                bracketCount--;

                if(bracketCount == 0){
                    parseNext=false;
                }
            }

            if(parseNext){
                //look flow
                matcher = lookUpFlowPatter.matcher(line);
                if (matcher.find()) {
                    String flow = matcher.group(1);
                    testSourceItemInfo.addFlowName(flow);

                } else {
                    matcher = lookUpFlowPatter2.matcher(line);
                    if (matcher.find()) {
                        String flow = matcher.group(1);
                        testSourceItemInfo.addFlowName(flow);
                    }
                }
            }
		}

		// Done with the file
		br.close();
		
		return testSourceItemInfo;
	}
}
