/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.model.module;

import org.mule.common.MuleVersion;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.Type;
import org.mule.devkit.model.module.rest.RestCall;

import java.util.List;

/**
 * The module interface represents a type in Java that has been annotated with
 * {@link Module}, {@link org.mule.api.annotations.Connector} or {@link org.mule.api.annotations.ExpressionLanguage}.
 */
public interface Module extends Type {

    boolean hasProcessorMethodWithParameter(Class<?> parameterType);

    boolean hasConfigurableWithType(Class<?> parameterType);

    /**
     * Get a list of all methods annotated with @Source
     * <p/>
     * This method will recursively go thru all superclasses as well.
     *
     * @return A list of all methods annotated with @Source
     * @see {@link SourceMethod}
     */
    List<SourceMethod> getSourceMethods();

    /**
     * Get a list of all methods annotated with @Processor
     * <p/>
     * This method will recursively go thru all superclasses as well.
     *
     * @return A list of all methods annotated with @Processor
     * @see {@link ProcessorMethod}
     */
    List<ProcessorMethod> getProcessorMethods();

    /**
     * Get a list of all methods annotated with @Filter
     * <p/>
     * This method will recursively go thru all superclasses as well.
     *
     * @return A list of all methods annotated with @Filter
     * @see {@link FilterMethod}
     */
    List<FilterMethod> getFilterMethods();

    /**
     * Retrieve a list of fields that have been annotated with {@link org.mule.api.annotations.Configurable}
     * <p/>
     * This method will recursively go thru all superclasses as well.
     *
     * @return A list of all fields annotated with {@link org.mule.api.annotations.Configurable}
     */
    List<Field> getConfigurableFields();

    /**
     * Retrieve a list of fields that have been annotated with {@link javax.inject.Inject}
     * <p/>
     * This method will recursively go thru all superclasses as well.
     *
     * @return A list of all fields annotated with {@link javax.inject.Inject}
     */
    List<Field> getInjectFields();

    /**
     * True if the module has at least one method annotated with {@link org.mule.api.annotations.Source}
     *
     * @return true if the module has at least one method annotated with {@link org.mule.api.annotations.Source}
     */
    boolean hasSources();

    /**
     * True if the module has at least one method annotated with {@link org.mule.api.annotations.Processor}
     *
     * @return true if the module has at least one method annotated with {@link org.mule.api.annotations.Processor}
     */
    boolean hasProcessors();

    /**
     * True if the module has at least one method annotated with {@link org.mule.api.annotations.Filter}
     *
     * @return true if the module has at least one method annotated with {@link org.mule.api.annotations.Filter}
     */
    boolean hasFilters();

    /**
     * True if the module has at least one method annotated with {@link org.mule.api.annotations.Transformer}
     *
     * @return true if the module has at least one method annotated with {@link org.mule.api.annotations.Transformer}
     */
    boolean hasTransformers();

    /**
     * Retrieve a list of all the methods inside the module that are annotated
     * with {@link org.mule.api.annotations.Transformer}
     *
     * @return A list with all the transformers
     */
    List<TransformerMethod> getTransformerMethods();

    /**
     * Retrieve a list of all the methods inside the module that are annotated
     * with {@link org.mule.api.annotations.TransformerResolver}
     *
     * @return A list with all the transformer resolvers
     */
    List<TransformerResolverMethod> getTransformerResolverMethods();

    boolean usesPooling();

    /**
     * Get the minimum Mule version supported by this module
     *
     * @return The minimum Mule version in string format
     */
    MuleVersion getMinMuleVersion();

    String getXmlNamespace();

    String getModuleName();

    String getAnnotatedSchemaLocation();

    String getVersionedSchemaLocation();

    String getCurrentSchemaLocation();

    String getModuleSchemaVersion();

    String getFriendlyName();

    /**
     * Get the description of this module
     *
     * @return The description
     */
    String getDescription();

    /**
     * Return true if this module requires a config element or false if it can work
     * without one.
     *
     * @return True if requires a config element, false otherwise
     */
    boolean needsConfig();

    ModuleKind getKind();

    /**
     * Retrieve the name of the config element
     */
    String getConfigElementName();

    /**
     * Retrieve the method that needs to be called during a particular lifecycle phase.
     *
     * @param phaseName The name of the phase. Supported phases are:
     *                  {@see org.mule.api.lifecycle.Startable.PHASE_NAME},
     *                  {@see org.mule.api.lifecycle.Stoppable.PHASE_NAME},
     *                  {@see org.mule.api.lifecycle.Initialisable.PHASE_NAME},
     *                  {@see org.mule.api.lifecycle.Disposable.PHASE_NAME}
     * @return The method responsible for the requested lifecycle phase.
     */
    Method getLifecycleMethod(String phaseName);

    /**
     * Retrieve the field annotated with {@link org.mule.api.annotations.rest.RestHttpClient}
     */
    Field getRestHttpClientField();

    /**
     * True if the module has at least one method annotated with {@link org.mule.api.annotations.rest.RestCall}
     *
     * @return true if the module has at least one method annotated with {@link org.mule.api.annotations.rest.RestCall}
     */
    boolean hasRestCalls();


    /**
     * True if the module has at least one method annotated with {@link org.mule.api.annotations.TransformerResolver}
     *
     * @return true if the module has at least one method annotated with {@link org.mule.api.annotations.TransformerResolver}
     */
    boolean hasTransformerResolver();

    /**
     * Retrieve a list of all the methods inside the module that are annotated
     * with {@link org.mule.api.annotations.rest.RestCall}
     *
     * @return A list with all the REST calls
     */
    List<RestCall> getRestCalls();

    /**
     * Retrieve the method annotated with {@link org.mule.api.annotations.MetaDataKeyRetriever}
     *
     *
     * @return Method
     */
    Method getMetaDataKeyRetrieverMethod();

    /**
     * Retrieve the method annotated with {@link org.mule.api.annotations.MetaDataRetriever}
     *
     *
     * @return Method
     */
    Method getMetaDataRetrieverMethod();

    /**
     * Check if the module is annotated with {@link org.mule.api.annotations.Connector}
     * @return
     */

    boolean isConnector();

    /**
     * Check if the module is annotated with {@link org.mule.api.annotations.Module}
     * @return
     */
    boolean isModule();

    boolean hasDynamicMetadata();

    public Module parentModule();

    public boolean hasMethodWithName(String methodName);


    boolean hasQueryTranslator();

    Method getQueryTranslatorMethod();
}