/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.api.annotations.oauth;

import org.mule.security.oauth.OnNoTokenPolicy;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * The class annotated with {@link OAuth2} is applied to an {@link org.mule.api.annotations.Connector} that uses the OAuth 2.0 a protocol for authentication.
 * Basic information about the Service Provider is required to go through the OAuth flow.
 */
@Target(ElementType.TYPE)
@Retention(RetentionPolicy.SOURCE)
@Documented
public @interface OAuth2 {

    /**
     * The URL defined by the Service Provider used to obtain an Access Token
     */
    String accessTokenUrl();

    /**
     * The URL defined by the Service Provider where the Resource Owner will be redirected to grant
     * authorization to the Consumer
     */
    String authorizationUrl();

    /**
     * A Java regular expression used to extract the verifier from the Service Provider response as a result
     * of the Resource Owner authorizing the Consumer.
     */
    String verifierRegex() default "code=([^&]+)";

    /**
     * A Java regular expression used to extract the Access Token from the Service Provider response.
     */
    String accessTokenRegex() default "\"access_token\":\"([^&]+?)\"";

    /**
     * A Java regular expression used to extract the expiration time of the Access Token (in seconds) from the
     * Service Provider response. If the this regular expression is not found in the Service Provider response
     * (whether the regular expression is wrong or the Access Token never expires), the Access Token will be
     * treated as if it would never expire.
     */
    String expirationRegex() default "\"expires_in\":([^&]+?),";

    /**
     * A Java regular expression used to extract the Refresh Token from the Service Provider response.
     */
    String refreshTokenRegex() default "\"refresh_token\":\"([^&]+?)\"";

    /**
     * Extra authorization parameters that will be appended as a query string arguments when building
     * the authorization URL.
     */
    OAuthAuthorizationParameter[] authorizationParameters() default {};

    OnNoTokenPolicy onNoTokenPolicy() default OnNoTokenPolicy.EXCEPTION;
}