/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.devkit.generation.api.gatherer.printing;

import com.sun.tools.javac.model.JavacElements;
import com.sun.tools.javac.tree.JCTree;
import com.sun.tools.javac.util.Pair;
import org.mule.devkit.generation.api.Context;
import org.mule.devkit.generation.api.gatherer.Notification;
import org.mule.devkit.generation.api.gatherer.NotificationGatherer;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.Element;
import javax.tools.Diagnostic;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.util.*;

public class GathererPrinterHelper implements PrinterGatherer{

    @Override
    public void printErrors(NotificationGatherer notificationGatherer, ProcessingEnvironment processingEnv, Context context) {
        printKind(notificationGatherer.getErrors(), processingEnv, context,  Diagnostic.Kind.ERROR, "error", "errors");
    }

    @Override
    public void printWarnings(NotificationGatherer notificationGatherer, ProcessingEnvironment processingEnv, Context context) {
        printKind(notificationGatherer.getWarnings(), processingEnv, context,  Diagnostic.Kind.WARNING, "warning", "warnings");
    }

    @Override
    public void printNotes(NotificationGatherer notificationGatherer, ProcessingEnvironment processingEnv, Context context) {
        printKind(notificationGatherer.getNotes(), processingEnv, context,  Diagnostic.Kind.NOTE, "note", "notes");
    }

    /**
     * <p>Given a list of {@link Notification}s, and a criteria to print them (see {@link Diagnostic.Kind}), it will iterate
     * over the list to describe while sorting them in a pretty way. If the list is empty, it will print nothing.</p>
     * <p>Sample of the output after calling this method for an ERROR scenario:
     *   [WARNING] ***************************************************
     *   [WARNING]
     *   [WARNING] File: /<path>/metadata-grouping-types-integration-tests/src/main/java/org/mule/devkit/it/grouping/BaseGroupingConnector.java
     *   [WARNING]    1) @Configurable fields are not allowed in @Connector/@Module class.
     *   [WARNING]    2) Another warning message
     *   [ERROR] ***************************************************
     *   [ERROR]The following (4) errors where encountered while compiling:
     *   [ERROR]
     *   [ERROR] File: /<path>/metadata-grouping-types-integration-tests/src/main/java/org/mule/devkit/it/grouping/BaseGroupingConnector.java
     *   [ERROR]    1) @Processor parameter cannot be arrays, use List instead.:28
     *   [ERROR] File: /<path>/integration-tests/metadata-grouping-types-integration-tests/src/main/java/org/mule/devkit/it/grouping/categories/Rest_V1_Category.java
     *   [ERROR]    2) Must have one only argument of type MetaDataKey.:32
     *   [ERROR]    4) @Configurable cannot be applied to field with static modifier.:27
     *   [ERROR] File: /<path>/integration-tests/metadata-grouping-types-integration-tests/src/main/java/org/mule/devkit/it/grouping/GroupingConnector.java
     *   [ERROR]    3) Another error Message
     *
     *   [ERROR] error on execute: An error ocurred while the DevKit was generating Java code. Check the logs for further details.
     * </p>
     * <p> This method ensures that, in any error scenario, the complete list will be printed in the user's console. This
     * means that, if an exception is thrown while sorting the errors and make them pretty (by invoking {@link #getCompleteNotificationLog(java.util.List, javax.annotation.processing.ProcessingEnvironment, String, String)})
     * the COMPLETE list of errors will be printed in the default way. </p>
     *
     * @param notificationsNotes the list of notifications to be printed while sorted
     * @param processingEnv the environment to process the file paths
     * @param context  the context needed to print the messages
     * @param kind the type of error we want to be displayed
     * @param singular the label used to describe the header if there is ONE notification in the list
     * @param plural the label used to describe the header if there is TWO (or more) notifications in the list
     */
    public void printKind(List<Notification> notificationsNotes, ProcessingEnvironment processingEnv, Context context, Diagnostic.Kind kind, String singular, String plural) {
        if( !notificationsNotes.isEmpty() ){
            try{
                List<String> notificationLog = getCompleteNotificationLog(notificationsNotes, processingEnv, singular, plural);
                for (String notification : notificationLog) {
                    context.getMessager().printMessage(kind, notification);
                }

            }catch (Exception e){
                //Making sure that, under any situation, the logs will be always displayed
                for(Notification notification : notificationsNotes){
                    Element element = notification.getDetails().getElement().unwrap();
                    context.getMessager().printMessage(kind, notification.getDetails().getMessage(), element);
                }
            }
        }
    }

    /**
     * Given a list of {@link Notification}s, it creates a map of ordered notifications grouped by file name.
     * Once done it, it constructs a builder with all the notifications printed in a cleaner way.
     *
     * @param notifications
     * @param processingEnv
     * @param plural
     * @param singular
     * @return a List of Strings with all the lines that must be printed
     * @throws IOException
     */
    private List<String> getCompleteNotificationLog(List<Notification> notifications, ProcessingEnvironment processingEnv, String singular, String plural) throws IOException {
        Integer index = 1;
        Map<String, List<FileNotificationMessage>> fileNotificationsMap = parseAndSortNotifications(notifications, processingEnv);
        List<String> notificationsList = new ArrayList<>();

        String specificMessage = notificationsList.size() > 1 ? "("+notificationsList.size()+") " + plural + " where" : singular + " was";
        notificationsList.add("***************************************************");
        notificationsList.add("The following " + specificMessage + " encountered while compiling:");
        notificationsList.add(" ");
        for (Map.Entry<String, List<FileNotificationMessage>> entry: fileNotificationsMap.entrySet()) {
            notificationsList.add("File: " + entry.getKey());
            for (FileNotificationMessage fnm : entry.getValue()) {
                notificationsList.add("   " + index + ") " + fnm.getMessage() + ":" + fnm.getLine());
                index++;
            }
        }
        return notificationsList;
    }


    /**
     * Given a list of {@link Notification}s, it parses them and grouped in a map by the following criteria:
     * <ol>
     *     <li>Each key is the file path, so for every notification it must be one entry</li>
     *     <li>Each value is a list of {@link FileNotificationMessage}, which holds the message as well as the line number where occurred</li>
     *     <li>Each value is ordered by the criteria of {@link FileNotificationMessage#compareTo(FileNotificationMessage)} </li>
     * </ol>
     *
     * @param notifications The list to be processed
     * @param processingEnv needed value to process all the list, and resolve the file paths
     * @return a map grouped by filepath, with its notifications gathered and sorted out
     * @throws IOException if there is an issue while reading the {@code lineNumberReader}
     */
    private Map<String, List<FileNotificationMessage>> parseAndSortNotifications(List<Notification> notifications, ProcessingEnvironment processingEnv) throws IOException {
        Map<String, List<FileNotificationMessage>> messageMap = new HashMap<String, List<FileNotificationMessage>>();

        List<FileNotificationMessage> fileNotificationMessageList = new ArrayList<FileNotificationMessage>();

        for(Notification notification : notifications){
            JavacElements elemUtils = (JavacElements) processingEnv.getElementUtils();
            Pair<JCTree, JCTree.JCCompilationUnit> treeTop = elemUtils.getTreeAndTopLevel(notification.getDetails().getElement().unwrap(), null, null);

            String file = treeTop.snd.getSourceFile().toUri().getPath();

            FileReader fr = new FileReader(file);
            LineNumberReader lineNumberReader = new LineNumberReader(fr);
            lineNumberReader.skip(treeTop.fst.getPreferredPosition());

            //the line number needs to be added a 1, because LineNumberReader is zero-index based
            int line= lineNumberReader.getLineNumber() +1;

            fileNotificationMessageList.add(new FileNotificationMessage(file, line, notification.getDetails().getMessage()));

            if(!messageMap.containsKey(file)){
                messageMap.put(file, new ArrayList<FileNotificationMessage>());
            }
            messageMap.get(file).add(new FileNotificationMessage(file, line, notification.getDetails().getMessage()));
        }

        for (Map.Entry<String, List<FileNotificationMessage>> entry: messageMap.entrySet()) {
            Collections.sort(entry.getValue());
        }
        return messageMap;
    }
}
