/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.tools.devkit.lic.security;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;

import javax.crypto.Cipher;
import java.io.*;
import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * @author MuleSoft, Inc
 */
public class KeyHandler {

    public static final String RSA = "RSA";
    public static final String RSA_ECB_TRANSFORMATION = "RSA/ECB/PKCS1Padding";
    public static final String SHA_512_WITH_RSA = "SHA512withRSA";

    public static void generatePair(String publicName, String privateName, int keySize) throws NoSuchAlgorithmException, IOException {

        KeyPairGenerator gen = KeyPairGenerator.getInstance(RSA);
        gen.initialize(keySize);
        KeyPair key = gen.generateKeyPair();
        PrivateKey privateKey = key.getPrivate();
        PublicKey publicKey = key.getPublic();

        // extract the encoded private key, this is an unencrypted PKCS#8 private key
        FileOutputStream pvt = new FileOutputStream(privateName);
        try {
            pvt.write(privateKey.getEncoded());
            pvt.flush();
        } finally {
            pvt.close();
        }

        FileOutputStream pub = new FileOutputStream(publicName);
        try {
            pub.write(publicKey.getEncoded());
            pub.flush();
        } finally {
            pub.close();
        }
    }

    public static PublicKey loadPub(String keyName, InputStream keystoreIS, char[] password) throws Exception {

        KeyStore keystore = KeyStore.getInstance(KeyStore.getDefaultType());
        keystore.load(keystoreIS, password);

        Key key = keystore.getKey(keyName, password);
        if (key instanceof PrivateKey) {
            return keystore.getCertificate(keyName).getPublicKey();
        }

        throw new RuntimeException("Public key "+ keyName +" not found");
    }

    public static PrivateKey loadPrv(String keyName, InputStream keystoreIS, char[] password) throws Exception {

        KeyStore keystore = KeyStore.getInstance(KeyStore.getDefaultType());
        keystore.load(keystoreIS, password);

        Key key = keystore.getKey(keyName, password);
        if (key instanceof PrivateKey) {
            return (PrivateKey) key;
        }

        throw new RuntimeException("Private key "+ keyName +" not found");
    }

    public static PrivateKey loadPrv(InputStream keyIS)
            throws NoSuchAlgorithmException, IOException, InvalidKeySpecException
    {
        PKCS8EncodedKeySpec pkcs8EncodedKeySpec = new PKCS8EncodedKeySpec(IOUtils.toByteArray(keyIS));
        return KeyFactory.getInstance(RSA).generatePrivate(pkcs8EncodedKeySpec);
    }

    public static PublicKey loadPub(InputStream keyIS)
            throws NoSuchAlgorithmException, IOException, InvalidKeySpecException
    {
        X509EncodedKeySpec x509EncodedKeySpec = new X509EncodedKeySpec(IOUtils.toByteArray(keyIS));
        return KeyFactory.getInstance(RSA).generatePublic(x509EncodedKeySpec);
    }

    public static byte[] encrypt(byte[] data, Key key) throws Exception {
        return encrypt(data, key, KeyHandler.RSA_ECB_TRANSFORMATION);
    }

    public static byte[] encrypt(byte[] data, Key key, String transformation) throws Exception {
        Cipher cipher = Cipher.getInstance(transformation);
        cipher.init(Cipher.ENCRYPT_MODE, key);

        return Base64.encodeBase64(cipher.doFinal(data));
    }

    public static byte[] decrypt(byte[] data, Key key) throws Exception {
        return decrypt(data, key, KeyHandler.RSA_ECB_TRANSFORMATION);
    }

    public static byte[] decrypt(byte[] data, Key key, String transformation) throws Exception {
        Cipher cipher = Cipher.getInstance(transformation);
        cipher.init(Cipher.DECRYPT_MODE, key);

        return cipher.doFinal(Base64.decodeBase64(data));
    }


}
