/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.maven.studio;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.*;
import org.apache.maven.project.MavenProjectHelper;
import org.codehaus.plexus.archiver.zip.ZipArchiver;
import org.eclipse.sisu.equinox.launching.internal.P2ApplicationLauncher;
import org.mule.devkit.generation.studio.SerializableArchiverPlanGenerator;
import org.mule.devkit.generation.studio.packaging.ArchiverPlan;
import org.mule.devkit.generation.studio.packaging.PackagePlanExecutor;
import org.mule.devkit.generation.studio.packaging.Placeholder;
import org.mule.devkit.generation.studio.packaging.SignComponent;
import org.mule.devkit.generation.utils.SerializationUtils;
import org.mule.devkit.maven.AbstractMuleMojo;

import java.io.File;
import java.io.IOException;

/**
 * Build a Mule plugin archive.
 */
@Mojo(name = "studio-package", defaultPhase = LifecyclePhase.PACKAGE, requiresDependencyResolution = ResolutionScope.RUNTIME)
public class StudioPackageMojo extends AbstractMuleMojo {

    @Component
    private MavenProjectHelper projectHelper;
    @Parameter(defaultValue= "${project.build.outputDirectory}", required = true)
    private File classesDirectory;

    @Parameter(defaultValue = "${keystorePath}")
    private String keystorePath;

    @Parameter(defaultValue = "${alias}")
    private String alias;

    @Parameter(defaultValue = "${storepass}")
    private String storepass;

    @Parameter(defaultValue = "${keypass}")
    private String keypass;

    @Parameter(defaultValue = "${tsaUrl}")
    private String tsaUrl;

    @Component
    private P2ApplicationLauncher launcher;

    @Component
    private P2ApplicationLauncher launcher2;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        if (skipStudioPluginPackage) {
            return;
        }

        ArchiverPlan archiverPlan = SerializationUtils.load(new File(classesDirectory, SerializableArchiverPlanGenerator.PACKAGE_DESCRIPTOR_PATH));

        SignComponent signComponent = new SignComponent(keystorePath, storepass, keypass, alias, tsaUrl);
        PackagePlanExecutor.RelativePathMap pathResolver = new PackagePlanExecutor.RelativePathMap(outputDirectory);
        PackagePlanExecutor packagePlanExecutor = new PackagePlanExecutor(pathResolver, signComponent);

        packagePlanExecutor.execute(archiverPlan);

        createContentAndArtifacts(pathResolver);

        /* Important: This is the name that the Nexus Update Site Plugin uses to find the artifact.
         * Be careful changing how the artifact will be found. */
        projectHelper.attachArtifact(project, "zip", "studio-plugin", buildZip(pathResolver));
    }

    private File buildZip(PackagePlanExecutor.RelativePathMap pathResolver) throws MojoExecutionException {
        ZipArchiver archiver = new ZipArchiver();

        File updateSiteDirectory = Placeholder.UPDATE_SITE_DIRECTORY.visit(pathResolver);

        File[] files = updateSiteDirectory.listFiles();

        if (files == null) {
            throw new MojoExecutionException(String.format("Files not found in [%s] (update site directory)", updateSiteDirectory));
        }

        for (File file : files) {
            if (file.isDirectory()) {
                archiver.addDirectory(file, file.getName() + "/");
            } else {
                archiver.addFile(file, file.getName());
            }
        }

        File destFile = new File(outputDirectory + "/UpdateSite.zip");
        archiver.setDestFile(destFile);

        // Remove destFile if already exists
        if (destFile.exists()) {
            destFile.delete();
        }

        try {
            archiver.createArchive();
        } catch (IOException e) {
            throw new MojoExecutionException("Error creating the update site zip file: ", e);
        }

        return destFile;
    }

    private void createContentAndArtifacts(PackagePlanExecutor.RelativePathMap pathResolver) throws MojoExecutionException {
        try {
            File updateSitePath = Placeholder.UPDATE_SITE_DIRECTORY.visit(pathResolver);
            File classesPath = Placeholder.CLASSES_DIRECTORY.visit(pathResolver);

            String updateSiteURI = updateSitePath.toURI().toString();
            File categoryFile = new File(classesPath, "category.xml").getAbsoluteFile();

            if (!categoryFile.exists()) {
                throw new MojoExecutionException("Error creating update site: category.xml does not exist");
            }

            String categoryXml = categoryFile.toURI().toString();

            int returnCode = featuresAndBundles(launcher, updateSitePath, updateSiteURI);
            if (returnCode != 0) {
                throw new MojoExecutionException("Error creating update site: Return code " + returnCode);
            }

            returnCode = categoryPublisher(launcher2, updateSiteURI, categoryXml);
            if (returnCode != 0) {
                throw new MojoExecutionException("Error creating update site: Return code " + returnCode);
            }

        } catch (Exception e) {
            throw new MojoExecutionException("Could not create update site", e);
        }

    }

    private int featuresAndBundles(P2ApplicationLauncher launcher, File updateSitePath, String updateSiteURI) {
        launcher.addArguments("-artifactRepository", updateSiteURI);
        launcher.addArguments("-metadataRepository", updateSiteURI);
        launcher.addArguments("-source", updateSitePath.getAbsolutePath());
        launcher.addArguments("-publishArtifacts", "-append");

        launcher.setApplicationName("org.eclipse.equinox.p2.publisher.FeaturesAndBundlesPublisher");

        return launcher.execute(20);
    }

    private int categoryPublisher(P2ApplicationLauncher launcher, String updateSiteURI, String categoryXml) {
        launcher.addArguments("-categoryDefinition", categoryXml);
        launcher.addArguments("-categoryQualifier");
        launcher.addArguments("-metadataRepository", updateSiteURI);

        launcher.setApplicationName("org.eclipse.equinox.p2.publisher.CategoryPublisher");
        return launcher.execute(20);
    }

}
