/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */

package org.mule.devkit.maven;

import org.apache.maven.model.License;
import org.apache.maven.plugin.logging.Log;
import org.mule.devkit.apt.MavenLicense;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class MavenLicenseBuilder {

    private static final List<String> POSSIBLE_LICENSE_FILE_NAMES = Arrays.asList("LICENSE.md", "license.txt", "License.md");

    private static final String LICENSE_DEFAULT_FILENAME = POSSIBLE_LICENSE_FILE_NAMES.get(0);

    private Log log;

    private File defaultLicenseFile;
    private List<File> possibleLicenseFiles = new ArrayList<File>();
    private List<License> pomFoundLicenses;

    public MavenLicenseBuilder(Log log, File basedir, List<License> licenses) {
        this.log = log;

        this.defaultLicenseFile = new File(basedir, LICENSE_DEFAULT_FILENAME);

        this.pomFoundLicenses = licenses;

        for (String licenseFileName : POSSIBLE_LICENSE_FILE_NAMES) {
            this.possibleLicenseFiles.add(new File(basedir, licenseFileName));
        }
    }

    public List<String> getPossibleLicenseFileNames() {
        return POSSIBLE_LICENSE_FILE_NAMES;
    }

    public MavenLicense build() {

        File licenseFile = null;
        String licenseBody = null;

        for (File possibleLicenseFile : this.possibleLicenseFiles) {
            if (possibleLicenseFile.exists()) {
                licenseFile = possibleLicenseFile;
                break;
            }
        }

        if (licenseFile != null) {
            try {
                licenseBody = readLicenseBodyFromFile(licenseFile);
            } catch (IOException e) {
                log.warn(String.format("License could not be read from %s: ", licenseFile) + e.getLocalizedMessage());
                log.debug("Falling back to reading license from pom.xml.");
            }
        }

        if (licenseBody == null) {
            MavenLicense license = licenseFromPom();

            if (license == null || license.getLicenseBody().equals("")) {
                log.error("No license will be used in the Studio plugin: No content was found at that location");
                return null;
            }

            FileWriter fileWriter = null;

            // After the first time fetching the URL save the license so
            // we don't need to download it again
            try {
                fileWriter = new FileWriter(this.defaultLicenseFile);
                org.apache.commons.io.IOUtils.copy(new StringReader(licenseBody), fileWriter);
            } catch (IOException e) {
                log.warn("No license file could be saved in: " + this.defaultLicenseFile);
            } finally {
                org.apache.commons.io.IOUtils.closeQuietly(fileWriter);
            }
        }

        String url = getLicenseURLFromPom();

        return new MavenLicense(licenseBody, url);
    }

    private String getLicenseURLFromPom() {

        if (this.pomFoundLicenses != null && this.pomFoundLicenses.size() > 0 && this.pomFoundLicenses.get(0) != null && this.pomFoundLicenses.get(0).getUrl() != null) {
            return this.pomFoundLicenses.get(0).getUrl();
        }

        return null;
    }

    private MavenLicense licenseFromPom() {
        if (pomFoundLicenses.size() < 1) {
            log.warn("No license will be used in the Studio plugin: No license section was found at pom.xml.");
            return null;
        }

        License mavenLicenseObject = pomFoundLicenses.get(0);

        if (mavenLicenseObject == null || mavenLicenseObject.getUrl() == null || mavenLicenseObject.getUrl().equals("")) {
            log.warn("No license will be used in the Studio plugin: No license URL was specified.");
            return null;
        }

        BufferedReader in = null;

        try {
            URL licenseURL = new URL(mavenLicenseObject.getUrl());

            in = new BufferedReader(
                    new InputStreamReader(licenseURL.openStream()));

            return new MavenLicense(org.apache.commons.io.IOUtils.toString(in), mavenLicenseObject.getUrl());

        } catch (IOException e) {
            log.warn("No license will be used in the Studio plugin: Error reading from the remote URL:  " + e.getLocalizedMessage());
            return null;
        } finally {
            org.apache.commons.io.IOUtils.closeQuietly(in);
        }
    }

    private String readLicenseBodyFromFile(File licensePath) throws IOException {
        FileReader licenseReader = null;
        try {
            licenseReader = new FileReader(licensePath);
            return org.apache.commons.io.IOUtils.toString(licenseReader);
        } finally {
            org.apache.commons.io.IOUtils.closeQuietly(licenseReader);
        }
    }
}
