/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.maven;

import org.mule.util.IOUtils;
import org.mule.util.StringUtils;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.DirectoryScanner;
import org.eclipse.egit.github.core.Blob;
import org.eclipse.egit.github.core.Commit;
import org.eclipse.egit.github.core.Reference;
import org.eclipse.egit.github.core.RepositoryId;
import org.eclipse.egit.github.core.Tree;
import org.eclipse.egit.github.core.TreeEntry;
import org.eclipse.egit.github.core.TypedResource;
import org.eclipse.egit.github.core.client.RequestException;
import org.eclipse.egit.github.core.service.DataService;
import org.eclipse.egit.github.core.util.EncodingUtils;

/**
 * Based on Maven plugin by Kevin Sawicki (kevin@github.com)
 */
@Mojo(name = "github-upload-doc")
public class GitHubDocMojo extends AbstractGitHubMojo {

    private static final String BRANCH_DEFAULT = "refs/heads/gh-pages";
    private static final int BUFFER_LENGTH = 8192;

    /**
     * Branch to update
     */
    @Parameter(defaultValue= "${branch}")
    private String branch = BRANCH_DEFAULT;

    /**
     * Path of tree
     */
    @Parameter(defaultValue= "${path}")
    private String path;

    /**
     * Commit message
     */
    @Parameter(defaultValue= "${message}")
    private String message;

    /**
     * Name of repository
     */
    @Parameter(defaultValue= "${github.repositoryName}")
    private String repositoryName;

    /**
     * Owner of repository
     */
    @Parameter(defaultValue= "${github.repositoryOwner}")
    private String repositoryOwner;

    /**
     * User name for authentication
     */
    @Parameter(defaultValue= "${github.userName}")
    private String userName;

    /**
     * User name for authentication
     */
    @Parameter(defaultValue = "${github.password}")
    private String password;

    /**
     * User name for authentication
     */
    @Parameter(defaultValue = "${github.oauth2Token}")
    private String oauth2Token;

    /**
     * Host for API calls
     */
    @Parameter(defaultValue = "${github.host}")
    private String host;

    /**
     * Paths and patterns to include
     */
    @Parameter(property = "github.includes")
    private String[] includes;

    /**
     * Removes an specific string from the output path. E.g. if you want to remove the target folder, you could call
     * with -Dgithub.removeFromPath=target/
     */
    @Parameter(property = "github.removeFromPath", defaultValue = "")
    private String removeFromPath;

    /**
     * Paths and patterns to exclude
     */
    @Parameter
    private String[] excludes = new String[]{"**/current.xml"};

    /**
     * Base directory to commit files from
     */
    @Parameter(property = "siteOutputDirectory", defaultValue = "${project.build.directory}", required = true)
    private File outputDirectory;

    /**
     * Project being built
     */
    @Parameter(defaultValue = "${project}", required = true)
    private MavenProject project;

    /**
     * Force reference update
     */
    @Parameter(defaultValue = "${github.force}")
    private boolean force;

    /**
     * Merge with existing the existing tree that is referenced by the commit
     * that the ref currently points to
     */
    @Parameter(defaultValue = "${github.merge}")
    private boolean merge;

    /**
     * Show what blob, trees, commits, and references would be created/updated
     * but don't actually perform any operations on the target GitHub
     * repository.
     */
    @Parameter(defaultValue = "${github.dryRun}")
    private boolean dryRun;

    /**
     * In case of failure of the mojo, how many times retry the execution
     */
    @Parameter(property= "github.retry.count", defaultValue = "3")
    private int retryCount;

    /**
     * The number of milliseconds to wait before retrying.
     */
    @Parameter(property= "github.sleep.time", defaultValue = "10000")
    private int sleepTime;

    /**
     * Whether to fail this mojo when it is not possible to upload the blob to GitHub
     */
    @Parameter(property = "github.fail.build", defaultValue = "false")
    private boolean failBuild;
    
    @Parameter(defaultValue = "${reactorProjects}")
    private List<MavenProject> reactorProjects;

    @Override
    public void execute() throws  MojoExecutionException {
        
        if (message == null) {
            message = "Updating documentation";
        }

        if (host == null) {
            host = "api.github.com";
        }

    	if (!project.getPackaging().equals("mule-module")){
    		if (isDebug()) {
    			debug(String.format("Skipping docs upload of project %s because packaging is %s", project.getName(), project.getPackaging()));
    		}
    		return;
    	}
    	
    	RepositoryId repository = getRepository(project, repositoryOwner, repositoryName);

        if (dryRun) {
            info("Dry run mode, repository will not be modified");
        }
        
        // Find files to include
        String baseDir = outputDirectory.getAbsolutePath();
        
        String[] includePaths = removeEmpties(includes);
        String[] excludePaths = removeEmpties(excludes);
        if (isDebug()) {
            debug("Includes: " + Arrays.toString(includePaths));
            debug(MessageFormat.format(
                    "Scanning {0} and including {1} and exluding {2}", baseDir,
                    Arrays.toString(includePaths),
                    Arrays.toString(excludePaths)));
        }
        String[] paths = getMatchingPaths(includePaths, excludePaths, baseDir);
        if (paths.length != 1) {
            info(MessageFormat.format("Creating {0} blobs", paths.length));
        } else {
            info("Creating 1 blob");
        }
        if (isDebug()) {
            debug(MessageFormat.format("Scanned files to include: {0}", Arrays.toString(paths)));
        }

        DataService service = new DataService(createClient(host, userName, password, oauth2Token));

        // Write blobs and build tree entries
        List<TreeEntry> entries = new ArrayList<TreeEntry>(paths.length);
        String prefix = path;
        if (prefix == null) {
            prefix = "";
        }
        if (prefix.length() > 0 && !prefix.endsWith("/")) {
            prefix += "/";
        }

        // Convert separator to forward slash '/'
        if ('\\' == File.separatorChar) {
            for (int i = 0; i < paths.length; i++) {
                paths[i] = paths[i].replace('\\', '/');
            }
        }

        for (String path : paths) {
            TreeEntry entry = new TreeEntry();

            //Evaluates removeFromPath for generating output path
            String entryPath = StringUtils.remove(path, removeFromPath);

            entry.setPath(prefix + entryPath);
            entry.setType(TreeEntry.TYPE_BLOB);
            entry.setMode(TreeEntry.MODE_BLOB);

            if (isDebug()) {
                debug("Generating entry for " + entry.getPath());
            }

            try {
                String blob = createBlob(service, repository, path);
                entry.setSha(blob);
            } catch (MojoExecutionException e) {
                if (failBuild) {
                    throw e;
                } else {
                    return;
                }
            }

            entries.add(entry);
        }

        Reference ref = null;
        try {
            ref = service.getReference(repository, branch);
        } catch (RequestException e) {
            if (404 != e.getStatus()) {
                throw new MojoExecutionException("Error getting reference: " + e.getMessage(), e);
            }
        } catch (IOException e) {
            throw new MojoExecutionException("Error getting reference: " + e.getMessage(), e);
        }

        if (ref != null && !TypedResource.TYPE_COMMIT.equals(ref.getObject().getType())) {
            throw new MojoExecutionException(
                    MessageFormat
                            .format("Existing ref {0} points to a {1} ({2}) instead of a commmit",
                                    ref.getRef(), ref.getObject().getType(),
                                    ref.getObject().getSha()));
        }

        // Write tree
        Tree tree;
        try {
            int size = entries.size();
            if (size != 1) {
                info(MessageFormat.format("Creating tree with {0} blob entries", size));
            } else {
                info("Creating tree with 1 blob entry");
            }
            String baseTree = null;
            if (merge && ref != null) {
                Tree currentTree = service.getCommit(repository,
                        ref.getObject().getSha()).getTree();
                if (currentTree != null) {
                    baseTree = currentTree.getSha();
                }
                info(MessageFormat.format("Merging with tree {0}", baseTree));
            }
            if (!dryRun) {
                tree = service.createTree(repository, entries, baseTree);
            } else {
                tree = new Tree();
            }
        } catch (IOException e) {
            throw new MojoExecutionException("Error creating tree: " + e.getMessage(), e);
        }

        // Build commit
        Commit commit = new Commit();
        commit.setMessage(message);
        commit.setTree(tree);

        // Set parent commit SHA-1 if reference exists
        if (ref != null) {
            commit.setParents(Collections.singletonList(new Commit().setSha(ref.getObject().getSha())));
        }

        Commit created;
        try {
            if (!dryRun) {
                created = service.createCommit(repository, commit);
            } else {
                created = new Commit();
            }
            info(MessageFormat.format("Creating commit with SHA-1: {0}", created.getSha()));
        } catch (IOException e) {
            throw new MojoExecutionException("Error creating commit: " + e.getMessage(), e);
        }

        TypedResource object = new TypedResource();
        object.setType(TypedResource.TYPE_COMMIT).setSha(created.getSha());
        if (ref != null) {
            // Update existing reference
            ref.setObject(object);
            try {
                info(MessageFormat.format(
                        "Updating reference {0} from {1} to {2}", branch,
                        commit.getParents().get(0).getSha(), created.getSha()));
                if (!dryRun) {
                    service.editReference(repository, ref, force);
                }
            } catch (IOException e) {
                throw new MojoExecutionException("Error editing reference: " + e.getMessage(), e);
            }
        } else {
            // Create new reference
            ref = new Reference().setObject(object).setRef(branch);
            try {
                info(MessageFormat.format(
                        "Creating reference {0} starting at commit {1}",
                        branch, created.getSha()));
                if (!dryRun) {
                    service.createReference(repository, ref);
                }
            } catch (IOException e) {
                throw new MojoExecutionException("Error creating reference: " + e.getMessage(), e);
            }
        }
    }
    
    private String createBlob(DataService service, RepositoryId repository, String path) throws MojoExecutionException {
        File file = new File(outputDirectory, path);
        long length = file.length();
        int size = length > Integer.MAX_VALUE ? Integer.MAX_VALUE : (int) length;
        ByteArrayOutputStream output = new ByteArrayOutputStream(size);
        FileInputStream stream = null;
        try {
            stream = new FileInputStream(file);
            byte[] buffer = new byte[BUFFER_LENGTH];
            int read;
            while ((read = stream.read(buffer)) != -1) {
                output.write(buffer, 0, read);
            }
        } catch (IOException e) {
            throw new MojoExecutionException("Error reading file: " + e.getMessage(), e);
        } finally {
            IOUtils.closeQuietly(stream);
        }

        Blob blob = new Blob().setEncoding(Blob.ENCODING_BASE64);
        blob.setContent(EncodingUtils.toBase64(output.toByteArray()));

        try {
            if (isDebug()) {
                debug(MessageFormat.format("Creating blob from {0}", file.getAbsolutePath()));
            }

            if (!dryRun) {
                return uploadBlobRetryIfError(service, repository, blob);
            } else {
                return null;
            }
        } catch (IOException e) {
            throw new MojoExecutionException("Error creating blob: " + e.getMessage(), e);
        }
    }

    private String uploadBlobRetryIfError(DataService service, RepositoryId repository, Blob blob) throws IOException, MojoExecutionException {
        for (int i = 0; i < retryCount; i++) {
            try {
                return service.createBlob(repository, blob);
            } catch (IOException e) {
                warn(String.format("Exception caught while uploading the documentation to GitHub, %s retries left", retryCount - i - 1), e);
                sleep(sleepTime);
            }
        }

        error("Cannot upload documentation to GitHub after retrying");
        throw new MojoExecutionException("Cannot upload documentation to GitHub after retrying");
    }

    /**
     * Create an array with only the non-null and non-empty values
     *
     * @param values
     * @return non-null but possibly empty array of non-null/non-empty strings
     */
    public static String[] removeEmpties(String... values) {
        if (values == null || values.length == 0) {
            return new String[0];
        }
        List<String> validValues = new ArrayList<String>();
        for (String value : values) {
            if (value != null && value.length() > 0) {
                validValues.add(value);
            }
        }
        return validValues.toArray(new String[validValues.size()]);
    }

    /**
     * Get matching paths found in given base directory
     *
     * @param includes
     * @param excludes
     * @param baseDir
     * @return non-null but possibly empty array of string paths relative to the
     *         base directory
     */
    public static String[] getMatchingPaths(String[] includes, String[] excludes, String baseDir) {
        DirectoryScanner scanner = new DirectoryScanner();
        scanner.setBasedir(baseDir);
        if (includes != null && includes.length > 0) {
            scanner.setIncludes(includes);
        }
        if (excludes != null && excludes.length > 0) {
            scanner.setExcludes(excludes);
        }
        scanner.scan();
        return scanner.getIncludedFiles();
    }

    private void sleep(int sleepTime) {
        try {
            Thread.sleep(sleepTime);
        } catch (InterruptedException e) {
            // ignore
        }
    }
}