/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.maven;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.model.Resource;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.FileUtils;
import org.codehaus.plexus.util.StringUtils;

import javax.tools.*;
import javax.tools.JavaCompiler.CompilationTask;
import java.io.File;
import java.io.PrintWriter;
import java.util.*;


public abstract class AbstractAnnotationProcessorMojo extends AbstractMuleMojo {

    @Parameter(defaultValue = "${plugin.artifacts}", readonly = true)
    private List<Artifact> pluginArtifacts;

    /**
     * Specify the directory where to place generated source files (same behaviour of -s option)
     */
    @Parameter(required = false)
    private File outputDirectory;

    /**
     * Additional compiler arguments
     */
    @Parameter(required = false)
    private String compilerArguments;

    /**
     * Controls whether or not the output directory is added to compilation
     */
    @Parameter(required = false)
    private Boolean addOutputDirectoryToCompilationSources;

    /**
     * Indicates whether the build will continue even if there are compilation errors; defaults to true.
     */
    @Parameter(required = true, defaultValue = "true", property = "annotation.failOnError")
    private Boolean failOnError = true;

    /**
     * Indicates whether the compiler output should be visible, defaults to true
     */
    @Parameter(required = true, defaultValue = "true", property= "annotation.outputDiagnostics")
    private boolean outputDiagnostics = true;

    /**
     * character encoding used by source files.
     */
    @Parameter(defaultValue = "${project.build.sourceEncoding}")
    private String encoding;

    /**
     * System properties set before processor invocation.
     */
    @SuppressWarnings("rawtypes")
    @Parameter(required = false)
    private Map systemProperties;

    /**
     * includes pattern
     */
    @Parameter
    private String[] includes;

    /**
     * excludes pattern
     */
    @Parameter
    private String[] excludes;

    public abstract File getDefaultOutputDirectory();

    protected abstract File getOutputClassDirectory();

    protected abstract String[] getProcessors();

    protected abstract void addCompileSourceRoot(MavenProject project, String dir);

    protected abstract Set<String> getClasspathElements(Set<String> result);

    private String buildProcessor() {
        if (getProcessors() == null || getProcessors().length == 0) {
            return null;
        }

        StringBuilder result = new StringBuilder();

        int i;
        for (i = 0; i < getProcessors().length - 1; ++i) {
            result.append(getProcessors()[i]).append(',');
        }

        result.append(getProcessors()[i]);
        return result.toString();
    }


    private String buildCompileClasspath() {

        Set<String> pathElements = new HashSet<String>();
        if (pluginArtifacts != null) {
            for (Artifact a : pluginArtifacts) {
                if ("compile".equalsIgnoreCase(a.getScope()) || "runtime".equalsIgnoreCase(a.getScope())) {
                    File f = a.getFile();
                    if (f != null) {
                        pathElements.add(a.getFile().getAbsolutePath());
                    }
                }
            }
        }

        getClasspathElements(pathElements);
        StringBuilder result = new StringBuilder();

        for (String elem : pathElements) {
            result.append(elem).append(File.pathSeparator);
        }
        return result.toString();
    }

    /**
     *
     */
    public void execute() throws MojoExecutionException {
        if ("pom".equalsIgnoreCase(project.getPackaging())) {
            return;
        }

        try {
            executeWithExceptionsHandled();
        } catch (Exception e1) {
            getLog().error("Error on execute: " + e1.getMessage());
            //if verboseLogging is set to true, or we hit a NPE, then we are going to show the full stack to the user
            if (verboseLogging
                    || e1.getMessage().equals("java.lang.NullPointerException")){
                getLog().error(e1);
                getLog().error("please, report the whole stack as a bug in DevKit under http://forum.mulesoft.org/mulesoft");
            }
            if (failOnError) {
                throw new MojoExecutionException(e1.getMessage(), e1);
            }
        }
    }

    @SuppressWarnings("unchecked")
    private void executeWithExceptionsHandled() throws Exception {
        if (outputDirectory == null) {
            outputDirectory = getDefaultOutputDirectory();
        }

        ensureOutputDirectoryExists();
        addOutputToSourcesIfNeeded();

        JavaCompiler compiler = ToolProvider.getSystemJavaCompiler();
        StandardJavaFileManager fileManager = compiler.getStandardFileManager(null, null, null);

        final String includesString = (includes == null || includes.length == 0) ? "**/*.java" : StringUtils.join(includes, ",");
        final String excludesString = (excludes == null || excludes.length == 0) ? null : StringUtils.join(excludes, ",");

        List<File> filesToCompile = new ArrayList<File>(100);
        for (Object sourceDirectory : project.getCompileSourceRoots()) {
            File directory = new File((String) sourceDirectory);
            if (directory.exists()) {
                if (!directory.equals(outputDirectory)) {
                    filesToCompile.addAll(FileUtils.getFiles(directory, includesString, excludesString));
                }
            }
        }

        if (filesToCompile.isEmpty()) {
            getLog().warn("no source file(s) detected! processor compilationTask will be skipped!");
            return;
        }

        List<String> options = new ArrayList<String>(10);

        options.add("-cp");
        options.add(buildCompileClasspath());
        options.add("-proc:only");

        String processor = buildProcessor();
        if (processor != null) {
            options.add("-processor");
            options.add(processor);
        } else {
            getLog().info("No processors specified. Using default discovery mechanism.");
        }
        options.add("-d");
        options.add(getOutputClassDirectory().getPath());

        options.add("-s");
        options.add(outputDirectory.getPath());

        if (encoding != null) {
            options.add("-encoding");
            options.add(encoding);
        }

        addCompilerArguments(options);

        setSystemProperties();

        CompilationTask compilationTask = compiler.getTask(
                new PrintWriter(System.out),
                fileManager,
                createDiagnosticListener(),
                options,
                null,
                fileManager.getJavaFileObjectsFromFiles(filesToCompile));

        // Perform the compilation compilationTask.
        if (!compilationTask.call()) {
            throw new Exception("An error ocurred while the DevKit was generating Java code. Check the logs for further details.");
        }
    }

    private void setSystemProperties() {
        if (systemProperties != null) {
            Set<Map.Entry<String, String>> pSet = systemProperties.entrySet();

            for (Map.Entry<String, String> e : pSet) {
                getLog().info(String.format("set system property : [%s] = [%s]", e.getKey(), e.getValue()));
                System.setProperty(e.getKey(), e.getValue());
            }

        }
    }

    private DiagnosticListener<JavaFileObject> createDiagnosticListener() {
        if (outputDiagnostics) {
            return new DiagnosticListener<JavaFileObject>() {
                @Override
                public void report(Diagnostic<? extends JavaFileObject> diagnostic) {
                    switch (diagnostic.getKind()) {
                        case ERROR:
                            getLog().error(diagnostic.toString().replace("error: ", "").replace("error on execute: ", ""));
                            break;
                        case WARNING:
                        case MANDATORY_WARNING:
                            getLog().warn(diagnostic.toString().replace("warning: ", ""));
                            break;
                        case NOTE:
                            String message = diagnostic.toString().replace("Note: ", "");
                            if (message.contains("DEBUG: ")) {
                                getLog().debug(message.replace("DEBUG: ", ""));
                            } else {
                                getLog().info(message);
                            }
                            break;
                        case OTHER:
                            getLog().debug(diagnostic.toString());
                            break;
                    }
                }
            };
        } else {
            return new DiagnosticListener<JavaFileObject>() {
                @Override
                public void report(Diagnostic<? extends JavaFileObject> diagnostic) {
                    // don't do anything since output diagnostics are disabled
                }
            };
        }
    }

    protected void addCompilerArguments(List<String> options) throws MojoExecutionException {
        if (!StringUtils.isEmpty(compilerArguments)) {
            for (String arg : compilerArguments.split(" ")) {
                if (!StringUtils.isEmpty(arg)) {
                    arg = arg.trim();
                    getLog().info("Adding compiler arg: " + arg);
                    options.add(arg);
                }
            }
        }
    }

    private void addOutputToSourcesIfNeeded() {
        final Boolean add = addOutputDirectoryToCompilationSources;
        if (add == null || add.booleanValue()) {
            getLog().info("Source directory: " + outputDirectory + " added");
            addCompileSourceRoot(project, outputDirectory.getAbsolutePath());
            getLog().info("Resource directory: " + outputDirectory + " added");

            Resource resourceDirectory = new Resource();
            resourceDirectory.setExcludes(Arrays.asList("**/*.java",
                    AnnotationProcessorMojo.LICENSE_DESCRIPTOR_FILE_NAME));
            resourceDirectory.setDirectory(outputDirectory.getAbsolutePath());
            project.addResource(resourceDirectory);
        }
    }

    private void ensureOutputDirectoryExists() {
        final File f = outputDirectory;
        if (!f.exists()) {
            f.mkdirs();
        }
        if (!getOutputClassDirectory().exists()) {
            getOutputClassDirectory().mkdirs();
        }
    }
}
