/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.maven;

import org.apache.maven.model.Dependency;
import org.apache.maven.model.Resource;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;
import org.mule.devkit.apt.AnnotationProcessorContext;
import org.mule.devkit.apt.MavenLicense;
import org.mule.devkit.generation.utils.SerializationUtils;

import java.io.File;
import java.io.Serializable;
import java.util.*;

@Mojo(name = "generate-sources", defaultPhase = LifecyclePhase.GENERATE_SOURCES, requiresDependencyResolution = ResolutionScope.COMPILE)
public class AnnotationProcessorMojo extends AbstractAnnotationProcessorMojo {

    private static String[] processors = {"org.mule.devkit.apt.AnnotationProcessor"};
    public static final String LICENSE_DESCRIPTOR_FILE_NAME = "license.descriptor";

    /**
     * project classpath
     */
    @Parameter(defaultValue = "${project.compileClasspathElements}", required = true, readonly = true)
    @SuppressWarnings("unchecked")
    private List classpathElements;

    @Parameter(defaultValue = "${project.build.directory}/generated-sources/mule", required = true)
    private File defaultOutputDirectory;

    /**
     *  Set the destination directory for class files (same behaviour of -d option)
     */
    @Parameter(required = false, defaultValue = "${project.build.outputDirectory}")
    private File outputClassDirectory;

    /**
     * Skip JavaDoc validation
     */
    @Parameter(required = false, property= "devkit.javadoc.check.skip", defaultValue = "true")
    private boolean skipJavaDocValidation;
    /**
     * Update site category
     */
    @Parameter(defaultValue = "${category}")
    private String category;

    @Override
    protected File getOutputClassDirectory() {
        return outputClassDirectory;
    }

    @Override
    protected String[] getProcessors() {
        return processors;
    }

    @Override
    protected void addCompileSourceRoot(MavenProject project, String dir) {
        project.addCompileSourceRoot(dir);
    }

    @Override
    public File getDefaultOutputDirectory() {
        return defaultOutputDirectory;
    }

    @Override
    @SuppressWarnings("unchecked")
    protected Set<String> getClasspathElements(Set<String> result) {
        List<Resource> resources = project.getResources();

        if (resources != null) {
            for (Resource r : resources) {
                result.add(r.getDirectory());
            }
        }

        result.addAll(classpathElements);

        return result;
    }

    @Override
    protected void addCompilerArguments(List<String> options) throws MojoExecutionException {
        addCompilerJavadocArgument(options);
        addCompilerStudioPluginArgument(options);
        addVerboseLoggingArguments(options);

        File mavenInformationFile = createMavenInformationFile();
        options.add("-A" + AnnotationProcessorContext.MAVEN_INFORMATION_FILE + "=" + mavenInformationFile);

        super.addCompilerArguments(options);
    }

    private File createMavenInformationFile() throws MojoExecutionException {
        File mavenInformationFile = new File(defaultOutputDirectory, "maven-information.descriptor");
        Map<String, Serializable> mavenInformation = initializeMavenInformation();

        mavenInformation.put("mavenLicense", createLicenseFile().getAbsolutePath());

        mavenInformation.put("mavenDependencies", getProjectDependenciesAsString());

        SerializationUtils.save(mavenInformationFile, mavenInformation);
        return mavenInformationFile;
    }

    private File createLicenseFile() throws MojoExecutionException {
        File licenseFile = new File(defaultOutputDirectory, LICENSE_DESCRIPTOR_FILE_NAME);

        MavenLicenseBuilder mavenLicenseBuilder = new MavenLicenseBuilder(this.getLog(), project.getBasedir(),
                project.getLicenses());

        MavenLicense mavenLicense = mavenLicenseBuilder.build();

        if (mavenLicense == null) {
            List<String> possibleLicenseFileNames = mavenLicenseBuilder.getPossibleLicenseFileNames();
            throw new MojoExecutionException(String.format("Error!  None of the license files were found in the root " +
                            "directory (%s). Check that you are running your goal online and that the URL in the licenses " +
                            "section of the pom.xml is properly configured.",
                    Arrays.toString(possibleLicenseFileNames.toArray(new String[possibleLicenseFileNames.size()]))));
        }

        SerializationUtils.save(licenseFile, mavenLicense);

        return licenseFile;
    }

    private ArrayList<String> getProjectDependenciesAsString() {
        List<Dependency> dependencies = project.getDependencies();

        ArrayList<String> dependenciesAsString = new ArrayList<String>();

        for ( Dependency dependency : dependencies ) {
            dependenciesAsString.add(dependency.getGroupId() + ":" + dependency.getArtifactId() + ":" +
                    dependency.getVersion() + ":" + dependency.getScope() + ":" + dependency.isOptional());
        }
        return dependenciesAsString;
    }

    private Map<String, Serializable> initializeMavenInformation() {
        Map<String, Serializable> mavenInformation = new HashMap<String, Serializable>();

        mavenInformation.put("mavenGroupId", project.getGroupId());
        mavenInformation.put("mavenArtifactId", project.getArtifactId());
        mavenInformation.put("mavenVersion", project.getVersion());
        mavenInformation.put("mavenName", project.getName());
        mavenInformation.put("mavenOutputDirectory", defaultOutputDirectory.getPath());
        mavenInformation.put("mavenBuildDirectory", project.getBasedir().getPath());
        mavenInformation.put("mavenCategory", category);

        return mavenInformation;
    }

    protected void addCompilerStudioPluginArgument(List<String> options) {
        if (skipStudioPluginPackage) {
            options.add("-AenabledStudioPluginPackage=false");
        } else {
            options.add("-AenabledStudioPluginPackage=true");
        }
    }

    protected void addCompilerJavadocArgument(List<String> options) {
        if (skipJavaDocValidation) {
            options.add("-AenableJavaDocValidation=false");
        } else {
            options.add("-AenableJavaDocValidation=true");
        }
    }

    protected void addVerboseLoggingArguments(List<String> options) {
        options.add("-A" + AnnotationProcessorContext.ENABLE_VERBOSE_LOGGING + "=" + verboseLogging);
    }
}