package org.mule.devkit.nexus;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.maven.index.ArtifactContext;
import org.apache.maven.index.ArtifactInfo;
import org.apache.maven.index.IndexerField;
import org.apache.maven.index.IndexerFieldVersion;
import org.apache.maven.index.context.IndexCreator;
import org.apache.maven.index.creator.AbstractIndexCreator;
import org.codehaus.plexus.component.annotations.Component;
import org.codehaus.plexus.logging.LogEnabled;
import org.codehaus.plexus.logging.Logger;
import org.mule.devkit.nexus.capabilities.DevkitNexusCapability;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.jar.JarFile;
import java.util.zip.ZipEntry;

@Component(role = IndexCreator.class, hint = PluginIndexCreator.ID)
public class PluginIndexCreator
        extends AbstractIndexCreator implements LogEnabled {
    public static final String ID = "mule-devkit-index";

    private Logger logger;

    public static final IndexerField FLD_IS_PLUGIN = new IndexerField(DevkitNexusCapability.IS_PLUGIN, IndexerFieldVersion.V3,
            "isDevKitPlugin", "Is DevKit Plugin?", Field.Store.YES, Field.Index.NOT_ANALYZED);

    public PluginIndexCreator() {
        super(ID);
    }

    public void enableLogging(final Logger logger) {
        this.logger = logger;
    }

    public void populateArtifactInfo(ArtifactContext artifactContext)
            throws IOException {
        ArtifactInfo ai = artifactContext.getArtifactInfo();

        File artifactFile = artifactContext.getArtifact();

        if (artifactFile != null && artifactFile.exists() &&
                (artifactFile.getName().endsWith(".jar") || artifactFile.getName().endsWith(".zip"))) {
            updateArtifactInfo(ai, artifactFile);
        }
    }

    public void updateDocument(ArtifactInfo ai, Document doc) {
        if (ai.getAttributes().containsKey(FLD_IS_PLUGIN.getKey())) {
            doc.add(FLD_IS_PLUGIN.toField(ai.getAttributes().get(FLD_IS_PLUGIN.getKey())));
        }
    }

    public boolean updateArtifactInfo(Document doc, ArtifactInfo artifactInfo) {
        int i = 0;
        if (doc.get(FLD_IS_PLUGIN.getKey()) != null) {
            artifactInfo.getAttributes().put(FLD_IS_PLUGIN.getKey(), doc.get(FLD_IS_PLUGIN.getKey()));
            i++;
        }

        return i > 0;
    }

    private void updateArtifactInfo(ArtifactInfo ai, File f)
            throws IOException {

        if (ai.classifier != null) {
            return;
        }

        JarFile jar = null;

        try {
            jar = new JarFile(f);

            ZipEntry pluginService = jar.getEntry("META-INF/services/org.mule.devkit.generation.api.Plugin");
            if (pluginService == null) {
                return;
            }
            logger.info("Marking " + ai.groupId + ":" + ai.artifactId + ":" + ai.classifier + ":" + ai.version + " as a DevKit plugin");
            ai.getAttributes().put(FLD_IS_PLUGIN.getKey(), Boolean.TRUE.toString());
        } finally {
            if (jar != null) {
                try {
                    jar.close();
                } catch (Exception e) {
                    getLogger().error("Could not close jar file properly.", e);
                }
            }
        }
    }

    @Override
    public String toString() {
        return ID;
    }

    public Collection<IndexerField> getIndexerFields() {
        return Arrays.asList(FLD_IS_PLUGIN);
    }
}