/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.expressionlanguage;

import org.mule.api.MuleContext;
import org.mule.api.MuleException;
import org.mule.api.MuleMessage;
import org.mule.api.annotations.ExpressionEvaluator;
import org.mule.api.annotations.ExpressionLanguage;
import org.mule.api.annotations.param.CorrelationGroupSize;
import org.mule.api.annotations.param.CorrelationId;
import org.mule.api.annotations.param.CorrelationSequence;
import org.mule.api.annotations.param.ExceptionPayload;
import org.mule.api.annotations.param.InboundHeaders;
import org.mule.api.annotations.param.InvocationHeaders;
import org.mule.api.annotations.param.MessageRootId;
import org.mule.api.annotations.param.MessageUniqueId;
import org.mule.api.annotations.param.OutboundHeaders;
import org.mule.api.annotations.param.Payload;
import org.mule.api.annotations.param.SessionHeaders;
import org.mule.api.context.MuleContextAware;
import org.mule.api.lifecycle.Disposable;
import org.mule.api.lifecycle.Initialisable;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.api.lifecycle.Startable;
import org.mule.api.lifecycle.Stoppable;
import org.mule.api.transformer.TransformerException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.utils.NameUtils;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.Parameter;
import org.mule.devkit.model.Type;
import org.mule.devkit.model.code.ExpressionFactory;
import org.mule.devkit.model.code.GeneratedCatchBlock;
import org.mule.devkit.model.code.GeneratedClass;
import org.mule.devkit.model.code.GeneratedConditional;
import org.mule.devkit.model.code.GeneratedField;
import org.mule.devkit.model.code.GeneratedInvocation;
import org.mule.devkit.model.code.GeneratedMethod;
import org.mule.devkit.model.code.GeneratedPackage;
import org.mule.devkit.model.code.GeneratedTry;
import org.mule.devkit.model.code.GeneratedVariable;
import org.mule.devkit.model.code.Modifier;
import org.mule.devkit.model.code.Op;
import org.mule.devkit.model.code.TypeReference;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.expression.ExpressionUtils;
import org.mule.devkit.processor.ExpressionEvaluatorSupport;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import javax.lang.model.type.TypeKind;

public class ExpressionEvaluatorGenerator extends AbstractExpressionLanguageGenerator {
    private final static List<Product> CONSUMES = Arrays.asList(Product.CAPABILITIES_ADAPTER,
            Product.LIFECYCLE_ADAPTER,
            Product.INJECTION_ADAPTER,
            Product.CONNECTION_IDENTIFIER_ADAPTER,
            Product.OAUTH_ADAPTER,
            Product.ABSTRACT_EXPRESSION_EVALUATOR);
    private final static List<Product> PRODUCES = Arrays.asList(Product.REGISTRY_BOOTSTRAP_ENTRY);

    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public boolean shouldGenerate(Module module) {
        return module.getKind() == ModuleKind.EXPRESSION_LANGUAGE && module.getMethodsAnnotatedWith(ExpressionEvaluator.class).size() > 0;
    }

    @Override
    public void generate(Module module) {
        String name = module.getAnnotation(ExpressionLanguage.class).name();

        Method<? extends Type> executableElement = module.getMethodsAnnotatedWith(ExpressionEvaluator.class).get(0);
        TypeReference moduleObject = ctx().getProduct(Product.CAPABILITIES_ADAPTER, module);
        GeneratedClass evaluatorClass = getEvaluatorClass(name, module);

        ctx().note("Generating expression evaluator " + evaluatorClass.fullName() + " for language at class " + module.getName());

        GeneratedField moduleField = generateModuleField(moduleObject, evaluatorClass);

        GeneratedMethod setMuleContext = evaluatorClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "setMuleContext");
        GeneratedVariable muleContext = setMuleContext.param(ref(MuleContext.class), "muleContext");
        GeneratedConditional ifModuleIsContextAware = setMuleContext.body()._if(Op._instanceof(moduleField, ref(MuleContextAware.class)));
        ifModuleIsContextAware._then().add(ExpressionFactory.cast(ref(MuleContextAware.class), moduleField).invoke("setMuleContext").arg(muleContext));

        GeneratedMethod start = evaluatorClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "start");
        start._throws(ref(MuleException.class));
        GeneratedConditional ifModuleIsStartable = start.body()._if(Op._instanceof(moduleField, ref(Startable.class)));
        ifModuleIsStartable._then().add(ExpressionFactory.cast(ref(Startable.class), moduleField).invoke("start"));

        GeneratedMethod stop = evaluatorClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "stop");
        stop._throws(ref(MuleException.class));
        GeneratedConditional ifModuleIsStoppable = stop.body()._if(Op._instanceof(moduleField, ref(Stoppable.class)));
        ifModuleIsStoppable._then().add(ExpressionFactory.cast(ref(Stoppable.class), moduleField).invoke("stop"));

        GeneratedMethod init = evaluatorClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "initialise");
        init._throws(ref(InitialisationException.class));
        GeneratedConditional ifModuleIsInitialisable = init.body()._if(Op._instanceof(moduleField, ref(Initialisable.class)));
        ifModuleIsInitialisable._then().add(ExpressionFactory.cast(ref(Initialisable.class), moduleField).invoke("initialise"));

        GeneratedMethod dispose = evaluatorClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "dispose");
        GeneratedConditional ifModuleIsDisposable = dispose.body()._if(Op._instanceof(moduleField, ref(Disposable.class)));
        ifModuleIsDisposable._then().add(ExpressionFactory.cast(ref(Disposable.class), moduleField).invoke("dispose"));

        generateConstructor(moduleObject, evaluatorClass, moduleField);

        generateGetName(name, evaluatorClass);

        generateSetName(evaluatorClass);

        GeneratedMethod evaluate = evaluatorClass.method(Modifier.PUBLIC, ref(Object.class), "evaluate");
        GeneratedVariable expression = evaluate.param(ref(String.class), "expression");
        GeneratedVariable message = evaluate.param(ref(MuleMessage.class), "message");

        GeneratedTry tryStatement = evaluate.body()._try();

        GeneratedInvocation newArray = ExpressionFactory._new(ref(Class.class).array());
        for (Parameter parameter : executableElement.getParameters()) {
            if (parameter.asTypeMirror().getKind() == TypeKind.BOOLEAN ||
                    parameter.asTypeMirror().getKind() == TypeKind.BYTE ||
                    parameter.asTypeMirror().getKind() == TypeKind.SHORT ||
                    parameter.asTypeMirror().getKind() == TypeKind.CHAR ||
                    parameter.asTypeMirror().getKind() == TypeKind.INT ||
                    parameter.asTypeMirror().getKind() == TypeKind.FLOAT ||
                    parameter.asTypeMirror().getKind() == TypeKind.LONG ||
                    parameter.asTypeMirror().getKind() == TypeKind.DOUBLE) {
                newArray.arg(ref(parameter.asTypeMirror()).boxify().staticRef("TYPE"));
            } else {
                newArray.arg(ref(parameter.asTypeMirror()).boxify().dotclass());
            }
            //argCount++;
        }
        GeneratedVariable parameterClasses = tryStatement.body().decl(ref(Class.class).array(), "parameterClasses", newArray);
        int argCount;
        GeneratedInvocation getMethod = moduleField.invoke("getClass").invoke("getMethod").arg(executableElement.getName()).arg(parameterClasses);
        GeneratedVariable moduleEvaluate = tryStatement.body().decl(ref(java.lang.reflect.Method.class), "evaluateMethod", getMethod);
        List<GeneratedVariable> types = new ArrayList<GeneratedVariable>();
        for (Parameter parameter : executableElement.getParameters()) {
            GeneratedVariable var = tryStatement.body().decl(ref(java.lang.reflect.Type.class), parameter.getName() + "Type", moduleEvaluate.invoke("getGenericParameterTypes").component(ExpressionFactory.lit(types.size())));
            types.add(var);
        }

        argCount = 0;
        GeneratedInvocation evaluateInvoke = moduleField.invoke(executableElement.getName());
        for (Parameter parameter : executableElement.getParameters()) {
            if (parameter.getAnnotation(Payload.class) != null) {
                evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(), ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(message.invoke("getPayload"))));
            } else if (parameter.getAnnotation(ExceptionPayload.class) != null) {
                evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(), ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(message.invoke("getExceptionPayload"))));
            } else if (parameter.getAnnotation(CorrelationId.class) != null) {
                evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(), ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(message.invoke("getCorrelationId"))));
            } else if (parameter.getAnnotation(CorrelationSequence.class) != null) {
                evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(), ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(message.invoke("getCorrelationSequence"))));
            } else if (parameter.getAnnotation(CorrelationGroupSize.class) != null) {
                evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(), ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(message.invoke("getCorrelationGroupSize"))));
            } else if (parameter.getAnnotation(MessageUniqueId.class) != null) {
                evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(), ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(message.invoke("getUniqueId"))));
            } else if (parameter.getAnnotation(MessageRootId.class) != null) {
                evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(), ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(message.invoke("getMessageRootId"))));
            } else if (parameter.asTypeMirror().toString().startsWith(MuleMessage.class.getName())) {
                evaluateInvoke.arg(message);
            } else if (parameter.getAnnotation(InboundHeaders.class) != null) {
                InboundHeaders inboundHeaders = parameter.getAnnotation(InboundHeaders.class);
                if (parameter.asType().isArrayOrList()) {
                    evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("INBOUND:" + inboundHeaders.value()).arg(message).arg(ref(List.class).dotclass())
                            )));
                } else if (parameter.asType().isMap()) {
                    evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("INBOUND:" + inboundHeaders.value()).arg(message).arg(ref(Map.class).dotclass())
                            )));
                } else {
                    evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("INBOUND:" + inboundHeaders.value()).arg(message)
                            )));
                }
            } else if (parameter.getAnnotation(SessionHeaders.class) != null) {
                SessionHeaders sessionHeaders = parameter.getAnnotation(SessionHeaders.class);
                if (parameter.asType().isArrayOrList()) {
                    evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("SESSION:" + sessionHeaders.value()).arg(message).arg(ref(List.class).dotclass())
                            )));
                } else if (parameter.asType().isMap()) {
                    evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("SESSION:" + sessionHeaders.value()).arg(message).arg(ref(Map.class).dotclass())
                            )));
                } else {
                    evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("SESSION:" + sessionHeaders.value()).arg(message)
                            )));
                }
            } else if (parameter.getAnnotation(OutboundHeaders.class) != null) {
                evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(),
                        ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("OUTBOUND:*").arg(message).arg(ref(Map.class).dotclass())
                        )));
            } else if (parameter.getAnnotation(InvocationHeaders.class) != null) {
                InvocationHeaders invocationHeaders = parameter.getAnnotation(InvocationHeaders.class);
                if (parameter.asType().isArrayOrList()) {
                    evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("INVOCATION:" + invocationHeaders.value()).arg(message).arg(ref(List.class).dotclass())
                            )));
                } else if (parameter.asType().isMap()) {
                    evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("INVOCATION:" + invocationHeaders.value()).arg(message).arg(ref(Map.class).dotclass())
                            )));
                } else {
                    evaluateInvoke.arg(ExpressionFactory.cast(ref(parameter.asTypeMirror()).boxify(),
                            ExpressionFactory.invoke("transform").arg(message).arg(types.get(argCount)).arg(
                                    ref(ExpressionUtils.class).staticInvoke("getPropertyWithScope").arg("INVOCATION:" + invocationHeaders.value()).arg(message)
                            )));
                }
            } else {
                if (parameter.asTypeMirror().toString().contains("String")) {
                    evaluateInvoke.arg(expression);
                }
            }
            argCount++;
        }

        tryStatement.body()._return(evaluateInvoke);

        catchAndRethrowAsRuntimeException(tryStatement, NoSuchMethodException.class);
        catchAndRethrowAsRuntimeException(tryStatement, TransformerException.class);
    }

    private void catchAndRethrowAsRuntimeException(GeneratedTry tryStatement, Class clazz) {
        GeneratedCatchBlock catchBlock = tryStatement._catch(ref(clazz));
        GeneratedVariable e = catchBlock.param("e");
        catchBlock.body()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg(e));
    }

    private GeneratedField generateModuleField(TypeReference typeElement, GeneratedClass evaluatorClass) {
        return evaluatorClass.field(Modifier.PRIVATE, typeElement, "module", ExpressionFactory._null());
    }

    private void generateConstructor(TypeReference typeElement, GeneratedClass evaluatorClass, GeneratedField module) {
        GeneratedMethod constructor = evaluatorClass.constructor(Modifier.PUBLIC);
        constructor.body().assign(module, ExpressionFactory._new(typeElement));
    }

    private void generateGetName(String name, GeneratedClass evaluatorClass) {
        GeneratedMethod getName = evaluatorClass.method(Modifier.PUBLIC, ref(String.class), "getName");
        getName.body()._return(ExpressionFactory.lit(name));
    }

    private void generateSetName(GeneratedClass evaluatorClass) {
        GeneratedMethod setName = evaluatorClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "setName");
        setName.param(ref(String.class), "name");
        setName.body()._throw(ExpressionFactory._new(ref(UnsupportedOperationException.class)));
    }

    private GeneratedClass getEvaluatorClass(String name, Module module) {
        GeneratedPackage pkg = ctx().getCodeModel()._package(module.getPackage().getName() + ExpressionLanguageNamingConstants.EXPRESSIONS_NAMESPACE);
        GeneratedClass evaluator = pkg._class(NameUtils.camel(name) + ExpressionLanguageNamingConstants.EXPRESSION_EVALUATOR_CLASS_NAME_SUFFIX, ExpressionEvaluatorSupport.class, new Class<?>[]{org.mule.api.expression.ExpressionEvaluator.class});
        evaluator._implements(ref(MuleContextAware.class));
        evaluator._implements(ref(Startable.class));
        evaluator._implements(ref(Stoppable.class));
        evaluator._implements(ref(Initialisable.class));
        evaluator._implements(ref(Disposable.class));

        ctx().registerProduct(Product.REGISTRY_BOOTSTRAP_ENTRY, module, evaluator.name(), evaluator);

        return evaluator;
    }

}