/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.javadoc;

import org.apache.commons.lang.StringUtils;
import org.mule.devkit.generation.api.Context;
import org.mule.devkit.generation.api.PostProcessor;
import org.mule.devkit.generation.api.PostProcessorException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.Type;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.util.IOUtils;
import org.springframework.beans.factory.xml.PluggableSchemaResolver;
import org.w3c.dom.Element;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ExamplesPostProcessor implements PostProcessor {

    private static final String JAXP_SCHEMA_LANGUAGE = "http://java.sun.com/xml/jaxp/properties/schemaLanguage";
    private static final String W3C_XML_SCHEMA = "http://www.w3.org/2001/XMLSchema";
    private static final String JAXP_SCHEMA_SOURCE = "http://java.sun.com/xml/jaxp/properties/schemaSource";
    private Map<String, String> examplesFilesByPath = new HashMap<String, String>();

    @Override
    public void postProcess(Context context) throws PostProcessorException {
        List<Module> modules = context.getModulesByProduct(Product.LIFECYCLE_ADAPTER);
        for (Module module : modules) {
            if (module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC) {
                List<Method<Type>> methodsToCheck = new ArrayList<Method<Type>>();
                methodsToCheck.addAll(module.getProcessorMethods());
                methodsToCheck.addAll(module.getSourceMethods());
                methodsToCheck.addAll(module.getTransformerMethods());
                File schema = getSchema(context, module);
                for (Method<Type> methodToCheck : methodsToCheck) {
                    String example = getExample(methodToCheck);
                    context.note("Validating example for method: " + methodToCheck.getName());
                    validateExampleAgainstSchema(module, methodToCheck, schema, example);
                }
            }
        }
    }

    private File getSchema(Context context, Module module) {
        String directory = context.getMavenInformation().getOutputDirectory();
        String schemaFileName = context.getSchemaModel().getSchemaLocationByNamespace(module.getXmlNamespace()).getFileName();
        return new File(directory, schemaFileName);
    }

    private String getExample(Method<Type> method) throws PostProcessorException {
        FileInputStream fileInputStream = null;
        try {
            String sampleTag = method.getJavaDocTagContent("sample.xml");
            String[] split = sampleTag.split(" ");
            String pathToExamplesFile = split[0];
            String exampleName = split[1];
            if (!examplesFilesByPath.containsKey(pathToExamplesFile)) {
                File examplesFile = getExamplesFile(method, pathToExamplesFile);
                fileInputStream = new FileInputStream(examplesFile);
                String examplesFileContent = IOUtils.toString(fileInputStream);
                examplesFilesByPath.put(pathToExamplesFile, examplesFileContent);
            }
            String examplesFileContents = examplesFilesByPath.get(pathToExamplesFile);
            return extractExampleFromFile(examplesFileContents, exampleName);
        } catch (IOException e) {
            throw new PostProcessorException(method, "Error extracting examples", e);
        } finally {
            IOUtils.closeQuietly(fileInputStream);
        }

    }

    private String extractExampleFromFile(String examplesFileContent, String exampleName) {
        String beginTag = String.format(JavaDocConstants.BEGIN_TAG, exampleName);
        String endTag = String.format(JavaDocConstants.END_TAG, exampleName);
        String example = examplesFileContent.substring(examplesFileContent.indexOf(beginTag) + beginTag.length(), examplesFileContent.indexOf(endTag));
        return StringUtils.trim(example);
    }

    private File getExamplesFile(Method<Type> method, String pathToExamplesFile) {
        String sourcePath = method.parent().getPathToSourceFile();
        int packageCount = StringUtils.countMatches(method.parent().getQualifiedName().toString(), ".") + 1;
        while (packageCount > 0) {
            sourcePath = sourcePath.substring(0, sourcePath.lastIndexOf("/"));
            packageCount--;
        }
        return new File(sourcePath, pathToExamplesFile);
    }

    protected void validateExampleAgainstSchema(Module module, Method method, File schema, String example) throws PostProcessorException {
        DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
        documentBuilderFactory.setNamespaceAware(true);
        documentBuilderFactory.setValidating(true);
        documentBuilderFactory.setAttribute(JAXP_SCHEMA_LANGUAGE, W3C_XML_SCHEMA);
        documentBuilderFactory.setAttribute(JAXP_SCHEMA_SOURCE, schema);

        try {
            DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
            documentBuilder.setErrorHandler(createErrorHandler());
            documentBuilder.setEntityResolver(new PluggableSchemaResolver(getClass().getClassLoader()));
            String exampleWithNamespace = addNamespaceToExample(module, example);
            documentBuilder.parse(IOUtils.toInputStream(exampleWithNamespace));
        } catch (SAXException e) {
            throw new PostProcessorException(method, "Error validating example: " + e.getMessage() + " Failing example: " + example, e);
        } catch (ParserConfigurationException e) {
            throw new PostProcessorException(method, "Error validating example: " + e.getMessage() + " Failing example: " + example, e);
        } catch (IOException e) {
            throw new PostProcessorException(method, "Error validating example: " + e.getMessage() + " Failing example: " + example, e);
        } catch (TransformerException e) {
            throw new PostProcessorException(method, "Error validating example: " + e.getMessage() + " Failing example: " + example, e);
        }
    }

    private ErrorHandler createErrorHandler() {
        return new ErrorHandler() {
            @Override
            public void warning(SAXParseException exception) throws SAXException {
                // nothing
            }

            @Override
            public void error(SAXParseException exception) throws SAXException {
                throw exception;
            }

            @Override
            public void fatalError(SAXParseException exception) throws SAXException {
                throw exception;
            }
        };
    }

    private String addNamespaceToExample(Module module, String example) throws SAXException, IOException, ParserConfigurationException, TransformerException {
        Element node = DocumentBuilderFactory
                .newInstance()
                .newDocumentBuilder()
                .parse(IOUtils.toInputStream(example))
                .getDocumentElement();
        node.setAttribute("xmlns:" + module.getModuleName(), module.getXmlNamespace());

        TransformerFactory transFactory = TransformerFactory.newInstance();
        Transformer transformer = transFactory.newTransformer();
        StringWriter buffer = new StringWriter();
        transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
        transformer.transform(new DOMSource(node), new StreamResult(buffer));
        return buffer.toString();
    }
}