/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */



package org.mule.devkit.generation.javadoc;

import org.mule.devkit.generation.api.AnnotationVerificationException;
import org.mule.devkit.generation.api.ModuleAnnotationVerifier;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.Identifiable;
import org.mule.devkit.model.Method;
import org.mule.devkit.model.Parameter;
import org.mule.devkit.model.Type;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.devkit.model.module.ProcessorMethod;
import org.mule.devkit.model.module.SourceMethod;
import org.mule.devkit.model.module.connectivity.ManagedConnectionModule;
import org.mule.util.IOUtils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.lang.model.type.TypeMirror;

import org.apache.commons.lang.StringUtils;

public class JavaDocAnnotationVerifier implements ModuleAnnotationVerifier {

    private Map<String, String> exampleFilesByName = new HashMap<String, String>();

    @Override
    public boolean shouldVerify(Module module) {
        return module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC;
    }

    @Override
    public void verify(Module module) throws AnnotationVerificationException {

        if (!hasComment(module)) {
            throw new AnnotationVerificationException(module, "Class " + module.getQualifiedName().toString() + " is not properly documented. A summary is missing.");
        }

        if (!module.hasJavaDocTag("author")) {
            throw new AnnotationVerificationException(module, "Class " + module.getQualifiedName().toString() + " needs to have an @author tag.");
        }

        for (Field variable : module.getConfigurableFields()) {
            if (!hasComment(variable)) {
                throw new AnnotationVerificationException(variable, "Field " + variable.getName() + " is not properly documented. The description is missing.");
            }
        }

        for (ProcessorMethod method : module.getProcessorMethods()) {
            validateMethod(module, method);
        }

        for (SourceMethod method : module.getSourceMethods()) {
            validateMethod(module, method);
        }

        for (Method method : module.getTransformerMethods()) {
            validateMethod(module, method);
        }

        if (module instanceof ManagedConnectionModule) {
            ManagedConnectionModule managedConnectionModule = (ManagedConnectionModule) module;
            validateAllParameters(managedConnectionModule.getConnectMethod());
            validateAllParameters(managedConnectionModule.getDisconnectMethod());
        }
    }

    private void validateAllParameters(Method<? extends Module> method) throws AnnotationVerificationException {
        for (Parameter variable : method.getParameters()) {
            if (!hasParameterComment(variable.getName(), variable.parent())) {
                throw new AnnotationVerificationException(variable, "Parameter " + variable.getName() + " of method " + method.getName() + " is not properly documented. A matching @param in the method documentation was not found. ");
            }
        }
    }

    private void validateMethod(Module module, Method method) throws AnnotationVerificationException {
        if (!hasComment(method)) {
            throw new AnnotationVerificationException(method, "Method " + method.getName() + " is not properly documented. A description of what it can do is missing.");
        }

        if (!method.getReturnType().toString().equals("void") &&
                !method.getReturnType().toString().contains("StopSourceCallback")) {
            if (!method.hasJavaDocTag("return")) {
                throw new AnnotationVerificationException(module, "The return type of a non-void method must be documented. Method " + method.getName() + " is at fault. Missing @return.");
            }
        }

        validateThrowComments(method);

        if (exampleDoesNotExist(method)) {
            throw new AnnotationVerificationException(module, "Method " + method.getName() + " does not have the example pointed by the {@sample.xml} tag");
        }

        validateAllParameters(method);
    }

    private void validateThrowComments(Method method) throws AnnotationVerificationException {
        List<? extends TypeMirror> thrownTypes = method.getThrownTypes();
        for(TypeMirror thrownType : thrownTypes) {
            String fullyQualifiedExceptionClassName = thrownType.toString();
            if(StringUtils.isEmpty(method.getThrowsComment(fullyQualifiedExceptionClassName))) {
                String exceptionClassName = fullyQualifiedExceptionClassName.substring(fullyQualifiedExceptionClassName.lastIndexOf(".") + 1);
                if(StringUtils.isEmpty(method.getThrowsComment(exceptionClassName))) {
                    throw new AnnotationVerificationException(method, "Every exception declared in the 'throws' clause needs to be documented. Method: " + method.getName() + " Exception: " + fullyQualifiedExceptionClassName);
                }
            }
        }
    }

    private boolean hasComment(Identifiable element) {
        String comment = element.getJavaDocSummary();
        return StringUtils.isNotBlank(comment);

    }

    private boolean hasParameterComment(String paramName, Identifiable element) {
        String comment = element.getJavaDocParameterSummary(paramName);
        return StringUtils.isNotBlank(comment);
    }

    protected boolean exampleDoesNotExist(Method<Type> method) throws AnnotationVerificationException {

        if (!method.hasJavaDocTag("sample.xml")) {
            throw new AnnotationVerificationException(method, "Method " + method.getName() + " does not contain an example using {@sample.xml} tag.");
        }

        String sample = method.getJavaDocTagContent("sample.xml");
        String[] split = sample.split(" ");

        if (split.length != 2) {
            throw new AnnotationVerificationException(method, "Check @sample.xml javadoc tag because is not well formed for method: " + method.getName());
        }

        String pathToExamplesFile = split[0];
        String exampleName = split[1];

        String sourcePath = method.parent().getPathToSourceFile();
        int packageCount = StringUtils.countMatches(method.parent().getQualifiedName().toString(), ".") + 1;
        while (packageCount > 0) {
            sourcePath = sourcePath.substring(0, sourcePath.lastIndexOf("/"));
            packageCount--;
        }

        boolean found = false;
        try {
            String examplesFileContent = getExamplesFile(pathToExamplesFile, sourcePath);
            String beginTag = String.format(JavaDocConstants.BEGIN_TAG, exampleName);
            String endTag = String.format(JavaDocConstants.END_TAG, exampleName);
            if (examplesFileContent.contains(beginTag) && examplesFileContent.contains(endTag)) {
                found = true;
            }
        } catch (IOException e) {
            found = false;
        }

        return !found;
    }

    private String getExamplesFile(String pathToExamplesFile, String sourcePath) throws FileNotFoundException {
        if(!exampleFilesByName.containsKey(pathToExamplesFile)) {
            File docFile = new File(sourcePath, pathToExamplesFile);
            FileInputStream inputStream = null;
            try {
                inputStream = new FileInputStream(docFile);
                exampleFilesByName.put(pathToExamplesFile, IOUtils.toString(inputStream));
            } finally {
                IOUtils.closeQuietly(inputStream);
            }
        }
        return exampleFilesByName.get(pathToExamplesFile);
    }
}