/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */


package org.mule.devkit.generation.javadoc;

import org.apache.commons.lang.StringUtils;
import org.mule.api.MuleEvent;
import org.mule.api.MuleMessage;
import org.mule.devkit.generation.api.AbstractBaseAnnotationVerifier;
import org.mule.devkit.generation.api.AnnotationVerificationException;
import org.mule.devkit.generation.api.gatherer.DevkitNotification;
import org.mule.devkit.generation.api.gatherer.Message;
import org.mule.devkit.generation.api.gatherer.NotificationGatherer;
import org.mule.devkit.model.*;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.devkit.model.module.ProcessorMethod;
import org.mule.devkit.model.module.SourceMethod;
import org.mule.devkit.model.module.connectivity.ManagedConnectionModule;
import org.mule.streaming.PagingConfiguration;
import org.mule.util.IOUtils;

import javax.lang.model.type.TypeMirror;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class JavaDocAnnotationVerifier extends AbstractBaseAnnotationVerifier {

    private Map<String, String> exampleFilesByName = new HashMap<String, String>();

    @Override
    public boolean shouldVerify(Module module) {
        return module.getKind() == ModuleKind.CONNECTOR || module.getKind() == ModuleKind.GENERIC;
    }

    @Override
    public void verify(Module module,NotificationGatherer gatherer) throws AnnotationVerificationException {
        setGatherer(gatherer);
        if (!hasComment(module)) {
            addMessage(module, Message.CLASS_MISSING_SUMMARY, module.getQualifiedName().toString());
        }

        if (!module.hasJavaDocTag("author")) {
            addMessage(module, Message.CLASS_MISSING_AUTHOR_TAG, module.getQualifiedName().toString());
        }

        for (Field variable : module.getConfigurableFields()) {
            if (!hasComment(variable)) {
                addMessage(variable, Message.FIELD_MISSING_DESCRIPTION, variable.getName());
            }
        }

        for (ProcessorMethod method : module.getProcessorMethods()) {
            validateMethod(module, method);
        }

        for (SourceMethod method : module.getSourceMethods()) {
            validateMethod(module, method);
        }

        for (Method method : module.getTransformerMethods()) {
            validateMethod(module, method);
        }

        if (module instanceof ManagedConnectionModule) {
            ManagedConnectionModule managedConnectionModule = (ManagedConnectionModule) module;
            validateAllParameters(managedConnectionModule.getConnectMethod());
            validateAllParameters(managedConnectionModule.getDisconnectMethod());
        }
    }

    private void validateAllParameters(Method method) throws AnnotationVerificationException {
        for (Parameter variable : (List<Parameter>)method.getParameters()) {
            boolean needsJavadoc = needsJavadoc(variable, method);
            if ( needsJavadoc &&
                    !hasParameterComment(variable.getName(), variable.parent())) {
                addMessage(variable, Message.METHOD_PARAMETER_MISSING_JAVADOC, variable.getName(), method.getName());
            }
        }
    }

    /**
     * Checks if the parameter really needs to be described, as for some scenarios (with @Processors) the documentation is no needed.
     * Samples of the above are injected values, as MuleEvent or MuleMessage. Also, for pagination, the PagingConfiguration
     * documentation is hardcoded.
     *
     * @param variable
     * @param method
     * @return false if the {@code variable} does not need a javadoc, true otherwise
     */
    private boolean needsJavadoc(Parameter variable, Method method) {
        if (method instanceof ProcessorMethod){
            ProcessorMethod processorMethod = (ProcessorMethod) (Method) method;
            //checking for [MuleEvent|MuleMessage]
            String className = variable.asTypeMirror().toString();
            if (className.startsWith(MuleMessage.class.getName())
                    || className.startsWith(MuleEvent.class.getName())){
                return false;
            }
            //checking for PagingConfiguration
            if (processorMethod.isPaged() && className.startsWith(PagingConfiguration.class.getName())){
                return false;
            };
        }
        return true;
    }

    private void validateMethod(Module module, Method method) throws AnnotationVerificationException {
        if (!hasComment(method)) {
            addMessage(method, Message.METHOD_MISSING_DESCRIPTION, method.getName());
        }

        if (!method.getReturnType().toString().equals("void") &&
                !method.getReturnType().toString().contains("StopSourceCallback")) {
            if (!method.hasJavaDocTag("return")) {
                addMessage(method, Message.METHOD_MISSING_RETURN_TYPE_DOCUMENTATION, method.getName());
            }
        }

        validateThrowComments(method);

        validateSampleTag(method);

        validateAllParameters(method);
    }

    private void validateThrowComments(Method method) throws AnnotationVerificationException {
        List<? extends TypeMirror> thrownTypes = method.getThrownTypes();
        for(TypeMirror thrownType : thrownTypes) {
            String fullyQualifiedExceptionClassName = thrownType.toString();
            if(StringUtils.isEmpty(method.getThrowsComment(fullyQualifiedExceptionClassName))) {
                String exceptionClassName = fullyQualifiedExceptionClassName.substring(fullyQualifiedExceptionClassName.lastIndexOf(".") + 1);
                if(StringUtils.isEmpty(method.getThrowsComment(exceptionClassName))) {
                    addMessage(method, Message.METHOD_MISSING_DOCUMENTATION_FOR_THROWN_EXCEPTIONS, method.getName(), fullyQualifiedExceptionClassName);
                }
            }
        }
    }

    private boolean hasComment(Identifiable element) {
        String comment = element.getJavaDocSummary();
        return StringUtils.isNotBlank(comment);

    }

    private boolean hasParameterComment(String paramName, Identifiable element) {
        String comment = element.getJavaDocParameterSummary(paramName);
        return StringUtils.isNotBlank(comment);
    }

    protected void validateSampleTag(Method<Type> method) throws AnnotationVerificationException {
        String sample = method.getJavaDocTagContent("sample.xml");
        String[] split = sample.split(" ");

        if (split.length != 2) {
            return;
        }

        String pathToExamplesFile = split[0];
        String exampleName = split[1];

        String sourcePath = method.parent().getPathToSourceFile();
        int packageCount = StringUtils.countMatches(method.parent().getQualifiedName().toString(), ".") + 1;
        while (packageCount > 0) {
            sourcePath = sourcePath.substring(0, sourcePath.lastIndexOf("/"));
            packageCount--;
        }

        boolean found = false;
        try {
            String examplesFileContent = getExamplesFile(pathToExamplesFile, sourcePath);
            String beginTag = String.format(JavaDocConstants.BEGIN_TAG, exampleName);
            String endTag = String.format(JavaDocConstants.END_TAG, exampleName);
            if (examplesFileContent.contains(beginTag) && examplesFileContent.contains(endTag)) {
                found = true;
            }
            if (!found){
                addMessage(method, Message.SAMPLE_PROCESSOR_XML_DOES_NOT_EXIST, exampleName, method.getName());
            }
        } catch (IOException e) {
            found = false;
            addMessage(method, Message.SAMPLE_FILE_CONTAINING_EXAMPLES_DOES_NOT_EXIST, pathToExamplesFile, method.getName());
        }


    }

    private String getExamplesFile(String pathToExamplesFile, String sourcePath) throws FileNotFoundException {
        if(!exampleFilesByName.containsKey(pathToExamplesFile)) {
            File docFile = new File(sourcePath, pathToExamplesFile);
            FileInputStream inputStream = null;
            try {
                inputStream = new FileInputStream(docFile);
                exampleFilesByName.put(pathToExamplesFile, IOUtils.toString(inputStream));
            } finally {
                IOUtils.closeQuietly(inputStream);
            }
        }
        return exampleFilesByName.get(pathToExamplesFile);
    }

    private void addMessage(Identifiable element,DevkitNotification message,Object... args) {
        getGatherer().error(element, message, args);
    }
}