/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */

package org.mule.devkit.generation.license;

import org.mule.api.annotations.licensing.RequiresEnterpriseLicense;
import org.mule.api.annotations.licensing.RequiresEntitlement;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.devkit.generation.api.Context;
import org.mule.devkit.model.code.ExpressionFactory;
import org.mule.devkit.model.code.GeneratedCatchBlock;
import org.mule.devkit.model.code.GeneratedClass;
import org.mule.devkit.model.code.GeneratedConditional;
import org.mule.devkit.model.code.GeneratedMethod;
import org.mule.devkit.model.code.GeneratedTry;
import org.mule.devkit.model.code.GeneratedVariable;
import org.mule.devkit.model.code.Modifier;
import org.mule.devkit.model.code.TypeReference;
import org.mule.devkit.model.module.Module;

import com.mulesource.licm.EnterpriseLicenseKey;
import com.mulesource.licm.LicenseKeyException;
import com.mulesource.licm.LicenseManagementFactory;
import com.mulesource.licm.LicenseManager;
import com.mulesource.licm.feature.Feature;
import com.mulesource.licm.feature.FeatureValidator;

import java.util.Calendar;
import java.util.Date;

import de.schlichtherle.license.LicenseContentException;

/**
 * <p>Generator of the method that validates the license requirement of a connector/module</p>
 *
 * <p>It first checks for an enterprise license. If an evaluation license is not accepted then it checks that the </p>
 * <p>installed license is not an evaluation license. If an entitlement is required then it checks for the entitlement</p>
 */
public class LicenseCheckerGenerator {

    private Context context;
    private RequiresEntitlement requiresEntitlement;
    private RequiresEnterpriseLicense requiresEnterpriseLicense;

    public LicenseCheckerGenerator(Context context, Module element) {
        this.context = context;
        this.requiresEntitlement = element.getAnnotation(RequiresEntitlement.class);
        this.requiresEnterpriseLicense = element.getAnnotation(RequiresEnterpriseLicense.class);
    }

    public static boolean requiresLicense(Module element)
    {
        return element.getAnnotation(RequiresEntitlement.class) != null || element.getAnnotation(RequiresEnterpriseLicense.class) != null;
    }

    public TypeReference ref(Class<?> clazz) {
        return context.getCodeModel().ref(clazz);
    }

    public LicenseCheckerGenerator addTo(GeneratedClass messageProcessorClass) {
        GeneratedMethod checkMuleLicense = messageProcessorClass.method(Modifier.PRIVATE, context.getCodeModel().VOID, "checkMuleLicense");
        checkMuleLicense.javadoc().add("Obtains the expression manager from the Mule context and initialises the connector. If a target object ");
        checkMuleLicense.javadoc().add(" has not been set already it will search the Mule registry for a default one.");
        checkMuleLicense.javadoc().addThrows(context.getCodeModel().ref(InitialisationException.class));
        GeneratedTry tryStatement = checkMuleLicense.body()._try();
        GeneratedVariable licenseManager = tryStatement.body().decl(ref(LicenseManager.class), "licenseManager",
                ref(LicenseManagementFactory.class).staticInvoke("getInstance").invoke("createLicenseManager").arg(ref(LicenseManagementFactory.class).staticRef("MULE_EE")));

        GeneratedVariable license = tryStatement.body().decl(ref(EnterpriseLicenseKey.class), "license", licenseManager.invoke("validate").arg(ref(LicenseManagementFactory.class).staticRef("MULE_EE")));
        checkExpirationDate(tryStatement, license);

        if (allowsEvaluation()) {
            tryStatement.body()._if(license.invoke("isEvaluation"))._then()._throw(ExpressionFactory._new(ref(RuntimeException.class)).
                    arg("This Module does not allow Evaluation Licenses."));

        }

        if ( requiresEntitlement != null )
        {
           GeneratedVariable feature = tryStatement.body().decl(ref(Feature.class), "feature", ExpressionFactory._new(ref(Feature.class)).arg(requiresEntitlement.name()).arg(requiresEntitlement.description()));
           tryStatement.body().add(ExpressionFactory._new(ref(FeatureValidator.class)).arg(feature).invoke("validate").arg(license));
            GeneratedCatchBlock catchBlock = tryStatement._catch(ref(LicenseContentException.class));
            catchBlock.body()._throw(ExpressionFactory._new(ref(RuntimeException.class)).
                    arg("This module requires " + requiresEntitlement.name() + " entitlement in your license"));
        }

        GeneratedCatchBlock catchBlock = tryStatement._catch(ref(LicenseKeyException.class));
        catchBlock.body()._throw(ExpressionFactory._new(ref(RuntimeException.class)).
                arg("This Module requires an Enterprise license"));

        return this;
    }

    private void checkExpirationDate(GeneratedTry tryStatement, GeneratedVariable license) {
        GeneratedVariable expirationDate = tryStatement.body().decl(ref(Calendar.class), "expirationDate", ref(Calendar.class).staticInvoke("getInstance"));

        GeneratedConditional ifExpirationDateNotNull = tryStatement.body()._if(license.invoke("getExpirationDate").isNotNull());
        ifExpirationDateNotNull._then().add(expirationDate.invoke("setTime").arg(license.invoke("getExpirationDate")));
        ifExpirationDateNotNull._then()._if(expirationDate.invoke("after").arg(ExpressionFactory._new(ref(Date.class))))._then()._throw(ExpressionFactory._new(ref(RuntimeException.class)).arg("Your license has expire"));
    }

    public void into(GeneratedMethod method){
        method.body().invoke("checkMuleLicense");
    }

    private boolean allowsEvaluation() {
        return requiresEnterpriseLicense != null && !requiresEnterpriseLicense.allowEval();
    }
}
