/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */

package org.mule.devkit.generation.oauth;

import org.mule.api.annotations.oauth.OAuthAccessToken;
import org.mule.api.annotations.oauth.OAuthAccessTokenSecret;
import org.mule.devkit.generation.api.AnnotationVerificationException;
import org.mule.devkit.model.Field;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.ModuleKind;
import org.mule.devkit.model.module.oauth.OAuthAuthorizationParameter;
import org.mule.devkit.model.module.oauth.OAuthModule;
import org.mule.devkit.model.module.oauth.OAuthVersion;
import org.mule.devkit.model.schema.SchemaTypeConversion;

public class OAuth2ClientAnnotationVerifier extends AbstractOAuthClientAnnotationVerifier {

    @Override
    public boolean shouldVerify(Module module) {
        return super.shouldVerify(module) &&
                module instanceof OAuthModule && ((OAuthModule) module).getOAuthVersion() == OAuthVersion.V2;
    }


    public void verify(OAuthModule module) throws AnnotationVerificationException {
        if (module.getKind() == ModuleKind.GENERIC) {
            throw new AnnotationVerificationException(module, "It is not possible to use OAuth support in @Module annotated classes, use @Connector instead");
        }
        if (module.getConsumerKeyField() == null) {
            throw new AnnotationVerificationException(module, "@OAuth2 class must contain a field annotated with @OAuthConsumerKey");
        }
        if (!module.getConsumerKeyField().hasGetter()) {
            throw new AnnotationVerificationException(module, "@OAuthConsumerKey-annotated field must have a public getter");
        }
        if (!module.getConsumerKeyField().hasSetter()) {
            throw new AnnotationVerificationException(module, "@OAuthConsumerKey-annotated field must have a public setter");
        }
        if (module.getConsumerSecretField() == null) {
            throw new AnnotationVerificationException(module, "@OAuth2 class must contain a field annotated with @OAuthConsumerSecret");
        }
        if (!module.getConsumerSecretField().hasGetter()) {
            throw new AnnotationVerificationException(module, "@OAuthConsumerSecret-annotated field must have a public getter");
        }
        if (!module.getConsumerSecretField().hasSetter()) {
            throw new AnnotationVerificationException(module, "@OAuthConsumerSecret-annotated field must have a public setter");
        }
        if (classHasMethodWithParameterAnnotated(module, OAuthAccessToken.class)) {
            throw new AnnotationVerificationException(module, "The use of @OAuthAccessToken in parameters in @Processor methods is no longer supported as of DevKit 3.3.1. Instead please declare a field in the @Connector with the annotation instead.");
        }
        if (classHasMethodWithParameterAnnotated(module, OAuthAccessTokenSecret.class)) {
            throw new AnnotationVerificationException(module, "The use of @OAuthAccessTokenSecret in parameters in @Processor methods is no longer supported as of DevKit 3.3.1. Instead please declare a field in the @Connector with the annotation instead.");
        }
        if (module.getAccessTokenField() == null) {
            throw new AnnotationVerificationException(module, "@OAuth2 class must contain a field annotated with @OAuthAccessToken");
        }
        if (!module.getAccessTokenField().hasGetter()) {
            throw new AnnotationVerificationException(module, "@OAuthAccessToken-annotated field must have a public getter");
        }
        if (!module.getAccessTokenField().hasSetter()) {
            throw new AnnotationVerificationException(module, "@OAuthAccessToken-annotated field must have a public setter");
        }

        // check types on the authorization parameters
        if (module.getAuthorizationParameters() != null) {
            for (OAuthAuthorizationParameter parameter : module.getAuthorizationParameters()) {
                if (!SchemaTypeConversion.isSupported(parameter.getType().asTypeMirror().toString()) &&
                        !parameter.getType().isEnum()) {
                    throw new AnnotationVerificationException(module, "The type " + parameter.getType().getName() + " is not a supported authorization parameter type.");
                }
            }
        }

        if (module.getCallbackParameters() != null) {
            for (Field field : module.getCallbackParameters()) {
                if (!field.hasSetter()) {
                    throw new AnnotationVerificationException(module, "The field " + field.getName() + " must have a setter to be annotated with @OAuthCallbackParameter.");
                }
            }
        }

        verifyPostAuthorizationMethod(module);
    }

}
