/**
 * Mule Development Kit
 * Copyright 2010-2012 (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * This software is protected under international copyright law. All use of this software is
 * subject to MuleSoft's Master Subscription Agreement (or other master license agreement)
 * separately entered into in writing between you and MuleSoft. If such an agreement is not
 * in place, you may not use the software.
 */


package org.mule.devkit.generation.oauth.process;

import org.mule.RequestContext;
import org.mule.api.MuleEvent;
import org.mule.api.construct.FlowConstructAware;
import org.mule.api.context.MuleContextAware;
import org.mule.api.lifecycle.Initialisable;
import org.mule.api.lifecycle.Startable;
import org.mule.devkit.generation.api.GenerationException;
import org.mule.devkit.generation.api.Product;
import org.mule.devkit.generation.api.annotations.JustOnce;
import org.mule.devkit.generation.oauth.AbstractOAuthAdapterGenerator;
import org.mule.devkit.generation.oauth.OAuthClientNamingConstants;
import org.mule.devkit.model.code.*;
import org.mule.devkit.model.module.Module;
import org.mule.devkit.model.module.oauth.OAuthModule;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

@JustOnce
public class DefaultRestoreAccessTokenCallbackGenerator extends AbstractOAuthAdapterGenerator {
    private final static List<Product> CONSUMES = Arrays.asList(Product.OAUTH_INTERFACES);
    private final static List<Product> PRODUCES = Arrays.asList(Product.DEFAULT_RESTORE_ACCESS_TOKEN_CALLBACK);

    @Override
    public List<Product> consumes() {
        return CONSUMES;
    }

    @Override
    public List<Product> produces() {
        return PRODUCES;
    }

    @Override
    public boolean shouldGenerate(Module module) {
        return module instanceof OAuthModule && ((OAuthModule) module).getUserIdentifierMethod() == null;
    }

    @Override
    public void generate(Module module) throws GenerationException {
        GeneratedClass callbackClass = getDefaultRestoreAccessTokenCallbackClass(module);

        GeneratedField messageProcessor = generateFieldForMessageProcessor(callbackClass, "messageProcessor");
        GeneratedField logger = generateLoggerField(callbackClass);
        GeneratedField hasBeenStarted = generateFieldForBoolean(callbackClass, "hasBeenStarted");
        GeneratedField hasBeenInitialized = generateFieldForBoolean(callbackClass, "hasBeenInitialized");
        GeneratedField accessToken = generateFieldForString(callbackClass, "restoredAccessToken");
        GeneratedField accessTokenSecret = generateFieldForString(callbackClass, "restoredAccessTokenSecret");

        GeneratedMethod constructor = callbackClass.constructor(Modifier.PUBLIC);
        constructor.body().assign(hasBeenStarted, ExpressionFactory.FALSE);
        constructor.body().assign(hasBeenInitialized, ExpressionFactory.FALSE);

        callbackClass.getter(messageProcessor);
        callbackClass.setter(messageProcessor);

        GeneratedMethod restoreAccessTokenMethod = callbackClass.method(Modifier.PUBLIC, ctx().getCodeModel().VOID, "restoreAccessToken");
        GeneratedVariable event = restoreAccessTokenMethod.body().decl(ref(MuleEvent.class), "event", ref(RequestContext.class).staticInvoke("getEvent"));

        GeneratedConditional ifMuleContextAware = restoreAccessTokenMethod.body()._if(Op._instanceof(messageProcessor, ref(MuleContextAware.class)));
        ifMuleContextAware._then().add(
                ExpressionFactory.cast(ref(MuleContextAware.class), messageProcessor).invoke("setMuleContext").arg(
                        ref(RequestContext.class).staticInvoke("getEventContext").invoke("getMuleContext")
                )
        );

        GeneratedConditional ifFlowConstructAware = restoreAccessTokenMethod.body()._if(Op._instanceof(messageProcessor, ref(FlowConstructAware.class)));
        ifFlowConstructAware._then().add(
                ExpressionFactory.cast(ref(FlowConstructAware.class), messageProcessor).invoke("setFlowConstruct").arg(
                        ref(RequestContext.class).staticInvoke("getEventContext").invoke("getFlowConstruct")
                )
        );

        GeneratedConditional ifNotInitialized = restoreAccessTokenMethod.body()._if(Op.not(hasBeenInitialized));
        GeneratedConditional ifInitialisable = ifNotInitialized._then()._if(Op._instanceof(messageProcessor, ref(Initialisable.class)));
        GeneratedTry tryToInitialize = ifInitialisable._then()._try();
        tryToInitialize.body().add(
                ExpressionFactory.cast(ref(Initialisable.class), messageProcessor).invoke("initialise")
        );
        GeneratedCatchBlock catchInitlize = tryToInitialize._catch(ref(Exception.class));
        GeneratedVariable exception = catchInitlize.param("e");
        catchInitlize.body().add(
                logger.invoke("error").arg(
                        exception.invoke("getMessage")
                ).arg(
                        exception
                )
        );
        ifNotInitialized._then().assign(hasBeenInitialized, ExpressionFactory.TRUE);

        GeneratedConditional ifNotStarted = restoreAccessTokenMethod.body()._if(Op.not(hasBeenStarted));
        GeneratedConditional ifStartable = ifNotStarted._then()._if(Op._instanceof(messageProcessor, ref(Startable.class)));
        GeneratedTry tryToStart = ifStartable._then()._try();
        tryToStart.body().add(
                ExpressionFactory.cast(ref(Startable.class), messageProcessor).invoke("start")
        );
        GeneratedCatchBlock catchStart = tryToStart._catch(ref(Exception.class));
        exception = catchStart.param("e");
        catchStart.body().add(
                logger.invoke("error").arg(
                        exception.invoke("getMessage")
                ).arg(
                        exception
                )
        );

        ifNotStarted._then().assign(hasBeenStarted, ExpressionFactory.TRUE);

        GeneratedTry tryProcess = restoreAccessTokenMethod.body()._try();
        tryProcess.body().assign(event,
                messageProcessor.invoke("process").arg(event)
        );
        tryProcess.body().assign(accessToken,
                event.invoke("getMessage").invoke("getInvocationProperty").arg("OAuthAccessToken")
        );
        tryProcess.body().assign(accessTokenSecret,
                event.invoke("getMessage").invoke("getInvocationProperty").arg("OAuthAccessTokenSecret")
        );
        GeneratedCatchBlock catchProcess = tryProcess._catch(ref(Exception.class));
        exception = catchProcess.param("e");
        catchProcess.body().add(
                logger.invoke("error").arg(
                        exception.invoke("getMessage")
                ).arg(
                        exception
                )
        );

        GeneratedMethod getAccessTokenMethod = callbackClass.method(Modifier.PUBLIC, ref(String.class), "getAccessToken");
        getAccessTokenMethod.body()._return(accessToken);
        GeneratedMethod getAccessTokenSecretMethod = callbackClass.method(Modifier.PUBLIC, ref(String.class), "getAccessTokenSecret");
        getAccessTokenSecretMethod.body()._return(accessTokenSecret);
    }

    private GeneratedClass getDefaultRestoreAccessTokenCallbackClass(Module module) {
        GeneratedPackage pkg = ctx().getCodeModel()._package(module.getPackage().getName() + OAuthClientNamingConstants.OAUTH_NAMESPACE);
        GeneratedClass clazz = pkg._class(OAuthClientNamingConstants.DEFAULT_RESTORE_ACCESS_TOKEN_CALLBACK_CLASS_NAME);
        clazz._implements((TypeReference) ctx().getProduct(Product.RESTORE_ACCESS_TOKEN_CALLBACK_INTERFACE));
        ctx().registerProduct(Product.DEFAULT_RESTORE_ACCESS_TOKEN_CALLBACK, clazz);

        return clazz;
    }
}
